<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/api_helpers.php';

$activePage = 'credit';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get API settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('sms_api_key', 'sms_api_secret', 'email_api_key', 'email_api_secret', 'mpesa_consumer_key', 'mpesa_consumer_secret', 'mpesa_passkey', 'mpesa_shortcode', 'enable_mpesa')");
$stmt->execute();
$api_settings = [];
while ($row = $stmt->fetch()) {
    $api_settings[$row['setting_key']] = $row['setting_value'];
}

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'website')");
$stmt->execute();
$store_settings = [];
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
}

$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';
$storeWebsite = $store_settings['website'] ?? 'Store Website';

// Retrieve store logo data
$logoData = null;
$logoName = '';
$stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
$stmt->execute();
$logo = $stmt->fetch();
if ($logo && $logo['setting_binary']) {
    $logoData = $logo['setting_binary'];
    $logoName = $logo['setting_value'];
}

// Get credit customers summary
$stmt = $pdo->prepare("
    SELECT 
        c.id,
        c.name,
        SUM(ct.amount) as total_credit,
        COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as total_paid,
        COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as total_waived,
        SUM(ct.amount) - (COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0)) as total_balance
    FROM customers c
    JOIN credit_transactions ct ON c.id = ct.customer_id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    GROUP BY c.id, c.name
    HAVING total_balance > 0
    ORDER BY total_balance DESC
");
$stmt->execute();
$creditCustomers = $stmt->fetchAll();

// Get recent credit transactions
$stmt = $pdo->prepare("
    SELECT 
        ct.*,
        s.invoice_number as invoice_no,
        s.created_at as date,
        c.name as customer_name,
        COALESCE(SUM(cp.amount), 0) as paid,
        ct.amount - COALESCE(SUM(cp.amount), 0) as balance,
        CASE 
            WHEN COALESCE(SUM(cp.amount), 0) >= ct.amount THEN 'Paid'
            WHEN COALESCE(SUM(cp.amount), 0) > 0 THEN 'Partially Paid'
            ELSE 'Pending'
        END as status
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    JOIN customers c ON ct.customer_id = c.id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    GROUP BY ct.id
    ORDER BY s.created_at DESC
    LIMIT 20
");
$stmt->execute();
$recentTransactions = $stmt->fetchAll();

// Calculate total credit amounts
$totalCredit = array_sum(array_column($creditCustomers, 'total_balance'));
$amountPaid = array_sum(array_column($creditCustomers, 'total_paid'));
$pendingAmount = array_sum(array_column($creditCustomers, 'total_balance'));

// Handle payment action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['payment_id']) || isset($_POST['payment_customer_id']))) {
    header('Content-Type: application/json');
    
    try {
    $userId = $_SESSION['user_id'];
    $paymentMethod = $_POST['payment_method'];
    $reference = $_POST['reference'] ?? '';
    $amount = floatval($_POST['amount']);
    $selectedInvoiceIds = isset($_POST['payment_invoice_ids']) ? array_map('intval', (array)$_POST['payment_invoice_ids']) : [];
        
        // Log the incoming request
        error_log("Payment request received - Amount: $amount, Method: $paymentMethod, Customer ID: " . ($_POST['payment_customer_id'] ?? 'N/A'));
        
    if (empty($selectedInvoiceIds)) {
            throw new Exception("Please select at least one unpaid invoice to pay.");
        }

        if (!isset($_POST['payment_customer_id']) || empty($_POST['payment_customer_id'])) {
            throw new Exception("Customer ID is required.");
        }

        $customerId = $_POST['payment_customer_id'];
        $pdo->beginTransaction();
        
        // Get customer details
        $stmt = $pdo->prepare("SELECT name FROM customers WHERE id = ?");
        $stmt->execute([$customerId]);
        $customerName = $stmt->fetchColumn();
        
        if (!$customerName) {
            throw new Exception("Customer not found");
        }
        
        // Get selected invoices with their balances
        $inClause = implode(',', array_fill(0, count($selectedInvoiceIds), '?'));
        $params = $selectedInvoiceIds;
        $sql = "
            SELECT 
                ct.id,
                ct.sale_id,
                ct.amount,
                   s.invoice_number,
                (SELECT COALESCE(SUM(amount), 0) FROM credit_payments WHERE credit_transaction_id = ct.id AND (is_waived = 0 OR is_waived IS NULL)) as paid_amount,
                   GROUP_CONCAT(p.name SEPARATOR ', ') as items
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN sale_items si ON s.id = si.sale_id
            LEFT JOIN products p ON si.product_id = p.id
            WHERE ct.customer_id = ? AND ct.id IN ($inClause)
            GROUP BY ct.id
            HAVING ct.amount - paid_amount > 0
        ";
        array_unshift($params, $customerId);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $credits = $stmt->fetchAll();
        
        if (empty($credits)) {
            throw new Exception("No valid unpaid invoices found");
        }
        
        // Calculate total outstanding balance
        $totalOutstanding = 0;
        foreach ($credits as $credit) {
            $balance = $credit['amount'] - $credit['paid_amount'];
            $totalOutstanding += $balance;
        }
        
        // Store the previous balance before processing payment
        $previous_balance = $totalOutstanding;
        
        // Validate payment amount
        if ($amount <= 0) {
            throw new Exception("Payment amount must be greater than 0");
        }
        
        if ($amount > $totalOutstanding) {
            throw new Exception("Payment amount cannot exceed total outstanding balance of " . number_format($totalOutstanding, 2));
        }
        
        // Process payment
        $remainingAmount = $amount;
        $paymentDetails = [];
        $paymentDebug = [
            'old_balance' => $totalOutstanding,
            'paid_amount' => $amount,
            'new_balance' => $totalOutstanding - $amount,
            'payment_details' => []
        ];
        
        foreach ($credits as $credit) {
            if ($remainingAmount <= 0) break;
            
            $balance = $credit['amount'] - $credit['paid_amount'];
            if ($balance <= 0) continue;
            
            $paymentAmount = min($remainingAmount, $balance);
            
            // Insert payment record
            $stmt = $pdo->prepare("
                INSERT INTO credit_payments 
                (credit_transaction_id, sale_id, customer_id, amount, payment_method, reference, created_by) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $credit['id'],
                $credit['sale_id'],
                $customerId,
                $paymentAmount,
                $paymentMethod,
                $reference,
                $userId
            ]);
            
            $paymentDetails[] = [
                'invoice_number' => $credit['invoice_number'],
                'paid_amount' => $paymentAmount,
                'items' => $credit['items']
            ];
            
            $paymentDebug['payment_details'][] = [
                'invoice' => $credit['invoice_number'],
                'amount' => $paymentAmount,
                'balance_before' => $balance,
                'balance_after' => $balance - $paymentAmount
            ];
            
            $remainingAmount -= $paymentAmount;
        }
        
        // Calculate remaining balance after payment
        $remaining_balance = $previous_balance - $amount;
        
        // Log the payment
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (user_id, action, details) 
            VALUES (?, 'credit_payment', ?)
        ");
        $logDetails = json_encode([
            'customer_id' => $customerId,
            'amount' => $amount,
            'payment_method' => $paymentMethod,
            'reference' => $reference,
            'invoices' => $paymentDetails
        ]);
        $stmt->execute([$userId, $logDetails]);
        
        $pdo->commit();
        
        $successMessage = "Payment of " . number_format($amount, 2) . " processed successfully";
        $_SESSION['success'] = $successMessage;
        $_SESSION['pay_debug'] = $paymentDebug;
        
        error_log("Payment successful - $successMessage");
        
        echo json_encode([
            'success' => true,
            'message' => $successMessage,
            'customer' => $customerName,
            'amount' => $amount,
            'payment_method' => $paymentMethod,
            'payment_details' => $paymentDetails,
            'previous_balance' => $previous_balance,
            'remaining_balance' => $remaining_balance,
            'debug' => $paymentDebug
        ]);
        exit;
        
        } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        $errorMessage = "Error processing payment: " . $e->getMessage();
        error_log("Payment error: $errorMessage");
        $_SESSION['error'] = $errorMessage;
        
                echo json_encode([
                    'success' => false,
            'message' => $errorMessage
                ]);
                exit;
    }
}

// Handle reminder action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reminder_id'])) {
    header('Content-Type: application/json');
    
    try {
        $reminderId = $_POST['reminder_id'];
        $reminderType = $_POST['reminder_type'];
        $status = ['success' => true, 'message' => ''];
        
        // Get credit transaction and customer details
        $stmt = $pdo->prepare("
            SELECT ct.*, c.name as customer_name, c.email, c.phone, 
                   s.invoice_number, s.created_at as sale_date,
                   ct.amount - COALESCE(SUM(cp.amount), 0) as remaining_balance
            FROM credit_transactions ct
            JOIN customers c ON ct.customer_id = c.id
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.id = ?
            GROUP BY ct.id
        ");
        $stmt->execute([$reminderId]);
        $credit = $stmt->fetch();
        
        if (!$credit) {
            throw new Exception("Credit transaction not found");
        }

        if ($credit['remaining_balance'] <= 0) {
            throw new Exception("No remaining balance for this transaction");
        }

        $remainingBalance = number_format($credit['remaining_balance'], 2);
        $message = "Dear {$credit['customer_name']}, this is a reminder that you have a pending payment of {$currency} {$remainingBalance} for invoice #{$credit['invoice_number']} dated " . date('M d, Y', strtotime($credit['sale_date'])) . ". Please visit our store to make the payment.";
        $subject = "Payment Reminder - {$storeName}";
        
        // Log the reminder attempt
        $logDetails = json_encode([
            'customer_id' => $credit['customer_id'],
            'customer_name' => $credit['customer_name'],
            'invoice_number' => $credit['invoice_number'],
            'remaining_balance' => $remainingBalance,
            'reminder_type' => $reminderType
        ]);
        
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (user_id, action, details) 
            VALUES (?, 'credit_reminder', ?)
        ");
        $stmt->execute([$_SESSION['user_id'], $logDetails]);
        
        $sentMessages = [];
        
        if ($reminderType === 'sms' || $reminderType === 'both') {
            if (empty($credit['phone'])) {
                $status['message'] .= "SMS not sent: Customer has no phone number configured. ";
            } else {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
                $stmt->execute();
                $smsSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                
                if (($smsSettings['enable_sms'] ?? '0') !== '1') {
                    $status['message'] .= "SMS not sent: SMS notifications are not enabled. ";
                } elseif (empty($smsSettings['sms_api_key']) || empty($smsSettings['sms_api_secret']) || empty($smsSettings['sms_sender_id'])) {
                    $status['message'] .= "SMS not sent: SMS settings are not properly configured. ";
                } else {
                    $result = send_sms($credit['phone'], $message, $pdo);
                    if ($result['success']) {
                        $sentMessages[] = 'SMS';
                    } else {
                        $status['message'] .= "SMS not sent: " . ($result['error'] ?? 'Unknown error') . ". ";
                    }
                }
            }
        }
        
        if ($reminderType === 'email' || $reminderType === 'both') {
            if (empty($credit['email'])) {
                $status['message'] .= "Email not sent: Customer has no email address configured. ";
            } else {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
                $stmt->execute();
                $emailSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                
                if (($emailSettings['enable_email'] ?? '0') !== '1') {
                    $status['message'] .= "Email not sent: Email notifications are not enabled. ";
                } elseif (empty($emailSettings['smtp_host']) || empty($emailSettings['smtp_port']) || 
                         empty($emailSettings['smtp_username']) || empty($emailSettings['smtp_password'])) {
                    $status['message'] .= "Email not sent: Email settings are not properly configured. ";
                } else {
                    $result = send_email($credit['email'], $subject, $message, $pdo);
                    if ($result['success']) {
                        $sentMessages[] = 'Email';
                    } else {
                        $status['message'] .= "Email not sent: " . ($result['error'] ?? 'Unknown error') . ". ";
                    }
                }
            }
        }
        
        if (!empty($sentMessages)) {
            $status['message'] = "Successfully sent " . implode(' and ', $sentMessages) . " reminder(s). " . $status['message'];
        } else {
            $status['success'] = false;
            if (empty($status['message'])) {
                $status['message'] = "Failed to send any reminders.";
            }
        }
        
        echo json_encode($status);
        exit;
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Reminder error: ' . $e->getMessage()
        ]);
        exit;
    }
}

// Handle waive action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['waive_customer_id'])) {
    header('Content-Type: application/json');
    
    try {
        $customerId = $_POST['waive_customer_id'];
        $amount = floatval($_POST['waive_amount']);
        $reason = $_POST['waive_reason'];
        $userId = $_SESSION['user_id'];
        $selectedInvoiceIds = isset($_POST['waive_invoice_ids']) ? array_map('intval', (array)$_POST['waive_invoice_ids']) : [];

        if (empty($selectedInvoiceIds)) {
            echo json_encode([
                'success' => false,
                'message' => "Please select at least one unpaid invoice to waive."
            ]);
            exit;
        }

        // Get customer details
        $stmt = $pdo->prepare("SELECT name FROM customers WHERE id = ?");
        $stmt->execute([$customerId]);
        $customerName = $stmt->fetchColumn();
        
        if (!$customerName) {
            echo json_encode([
                'success' => false,
                'message' => "Customer not found"
            ]);
            exit;
        }

        // Only fetch and process selected invoices
        $inClause = implode(',', array_fill(0, count($selectedInvoiceIds), '?'));
        $params = $selectedInvoiceIds;
        $sql = "
            SELECT ct.id, ct.sale_id, ct.amount, 
                   (SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ct.id) as paid,
                   s.invoice_number,
                   GROUP_CONCAT(p.name SEPARATOR ', ') as items
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN sale_items si ON s.id = si.sale_id
            LEFT JOIN products p ON si.product_id = p.id
            WHERE ct.customer_id = ? AND ct.id IN ($inClause)
            GROUP BY ct.id
            HAVING ct.amount - paid > 0
        ";
        array_unshift($params, $customerId);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $credits = $stmt->fetchAll();

        // Calculate old balance BEFORE waiving
        $old_balance = 0;
        foreach ($credits as $credit) {
            $itemBalance = $credit['amount'] - $credit['paid'];
            if ($itemBalance > 0) {
                $old_balance += $itemBalance;
            }
        }

        // Validate
        if ($amount < 1) {
            echo json_encode([
                'success' => false,
                'message' => "Minimum waive amount is 1"
            ]);
            exit;
        }

        if ($amount > $old_balance) {
            echo json_encode([
                'success' => false,
                'message' => "Waive amount cannot exceed outstanding balance of selected invoices."
            ]);
            exit;
        }

        $new_balance = $old_balance - $amount;
        $waive_details = [];
        $waive_debug_details = [];
        $remainingAmount = $amount;
        $seen = [];

        $pdo->beginTransaction();

        foreach ($credits as $credit) {
            if (in_array($credit['id'], $seen)) continue;
            $seen[] = $credit['id'];
            $itemBalance = $credit['amount'] - $credit['paid'];
            if ($itemBalance > 0 && $remainingAmount > 0) {
                $waiveNow = min($remainingAmount, $itemBalance);
                $stmtWaive = $pdo->prepare("
                    INSERT INTO credit_payments 
                    (credit_transaction_id, sale_id, customer_id, amount, payment_method, reference, created_by, is_waived, waive_reason) 
                    VALUES (?, ?, ?, ?, 'cash', ?, ?, 1, ?)
                ");
                $stmtWaive->execute([
                    $credit['id'],
                    $credit['sale_id'],
                    $customerId,
                    $waiveNow,
                    'waiveD-' . date('YmdHis'),
                    $userId,
                    $reason
                ]);
                $waive_details[] = [
                    'invoice_number' => $credit['invoice_number'],
                    'waived_amount' => $waiveNow,
                    'items' => $credit['items']
                ];
                $waive_debug_details[] = 'Invoice ' . $credit['invoice_number'] . ': Waived ' . $waiveNow;
                $remainingAmount -= $waiveNow;
            }
        }

        $pdo->commit();

        // After waiving, recalculate the sum of balances for all invoices for this customer
        $stmt = $pdo->prepare("
            SELECT ct.amount, (SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ct.id) as paid
            FROM credit_transactions ct
            WHERE ct.customer_id = ?
        ");
        $stmt->execute([$customerId]);
        $all_credits = $stmt->fetchAll();
        $remaining_balance = 0;
        foreach ($all_credits as $c) {
            $bal = $c['amount'] - $c['paid'];
            if ($bal > 0.001) $remaining_balance += $bal;
        }

        echo json_encode([
            'success' => true,
            'message' => 'Amount waived successfully',
            'old_balance' => $old_balance,
            'waived_amount' => $amount,
            'new_balance' => $new_balance,
            'waive_details' => $waive_details,
            'remaining_balance' => $remaining_balance
        ]);
        exit;

    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        echo json_encode([
            'success' => false,
            'message' => "Error waiving amount: " . $e->getMessage()
        ]);
        exit;
    }
}

// Get filter parameters
$status = $_GET['status'] ?? '';
$customer_id = $_GET['customer_id'] ?? '';
$search = $_GET['search'] ?? '';

// Build query
$query = "SELECT ct.*, s.invoice_number, s.created_at as sale_date, 
          c.name as customer_name,
          GROUP_CONCAT(p.name SEPARATOR ', ') as items,
          SUM(si.quantity) as total_quantity,
          COALESCE(SUM(cp.amount), 0) as amount_paid,
          s.payment_method as sale_payment_method,
          s.paid_amount as sale_paid_amount,
          GROUP_CONCAT(DISTINCT sp.method SEPARATOR ', ') as payment_methods,
          GROUP_CONCAT(DISTINCT sp.amount SEPARATOR ', ') as payment_amounts,
          CASE 
              WHEN EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id) THEN 'paid'
              ELSE 'pending'
          END as status
          FROM credit_transactions ct
          JOIN sales s ON ct.sale_id = s.id
          JOIN customers c ON ct.customer_id = c.id
          LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
          LEFT JOIN sale_items si ON s.id = si.sale_id
          LEFT JOIN products p ON si.product_id = p.id
          LEFT JOIN sale_payments sp ON s.id = sp.sale_id
          WHERE 1=1";

$params = [];

if ($status) {
    if ($status === 'paid') {
        $query .= " AND EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id)";
    } else if ($status === 'pending') {
        $query .= " AND NOT EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id)";
    }
}

if ($customer_id) {
    $query .= " AND ct.customer_id = ?";
    $params[] = $customer_id;
}

if ($search) {
    $query .= " AND (s.invoice_number LIKE ? OR c.name LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$query .= " GROUP BY ct.id ORDER BY s.created_at DESC";

// Get credit transactions
$stmt = $pdo->prepare($query);
$stmt->execute($params);
$credits = $stmt->fetchAll();

// Get customers for filter
$stmt = $pdo->prepare("SELECT id, name FROM customers ORDER BY name");
$stmt->execute();
$customers = $stmt->fetchAll();

// Calculate statistics
$total_credit = 0;
$total_paid = 0;
$total_pending = 0;
$total_customers = 0;
$customers_with_credit = [];

// Get all statistics in a single query
$stmt = $pdo->prepare("
    WITH credit_summary AS (
        SELECT 
            ct.customer_id,
            ct.amount as original_amount,
            COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as amount_paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as amount_waived,
            ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as remaining_balance
        FROM credit_transactions ct
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        GROUP BY ct.id, ct.customer_id
        HAVING remaining_balance > 0
    )
    SELECT 
        COUNT(DISTINCT customer_id) as total_customers,
        SUM(remaining_balance) as total_credit,
        SUM(amount_paid) as total_paid,
        SUM(amount_waived) as total_waived,
        SUM(remaining_balance) as total_pending
    FROM credit_summary
");
$stmt->execute();
$totals = $stmt->fetch(PDO::FETCH_ASSOC);

$total_customers = $totals['total_customers'] ?? 0;
$total_credit = $totals['total_credit'] ?? 0;
$total_paid = $totals['total_paid'] ?? 0;
$total_pending = $totals['total_pending'] ?? 0;

// Group credits by customer
$credits_by_customer = [];
$cleared_credits_by_customer = [];
foreach ($credits as $credit) {
    $cid = $credit['customer_id'];
    // Calculate paid and waived for this transaction
    $stmtPaid = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND (is_waived = 0 OR is_waived IS NULL)");
    $stmtPaid->execute([$credit['id']]);
    $paid = $stmtPaid->fetchColumn();
    $waiveStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
    $waiveStmt->execute([$credit['id']]);
    $waived = $waiveStmt->fetchColumn();
    $balance = $credit['amount'] - $paid - $waived;

    // Only include invoices with a positive balance
    if ($balance > 0.001) {
        if (!isset($credits_by_customer[$cid])) {
            $credits_by_customer[$cid] = [
                'customer_name' => $credit['customer_name'] ?? $credit['name'] ?? 'Unknown',
                'total_credit' => 0,
                'total_paid' => 0,
                'total_waived' => 0,
                'total_balance' => 0,
                'status' => 'Not Paid',
                'transactions' => []
            ];
        }
        $credits_by_customer[$cid]['total_credit'] += $credit['amount'];
        $credits_by_customer[$cid]['total_paid'] += $paid;
        $credits_by_customer[$cid]['total_waived'] += $waived;
        $credits_by_customer[$cid]['total_balance'] += $balance;
        $credit['amount_paid'] = $paid;
        $credit['amount_waived'] = $waived;
        $credit['balance'] = $balance;
        $credits_by_customer[$cid]['transactions'][] = $credit;
    }
}
// Set status for each customer
foreach ($credits_by_customer as &$group) {
    if ($group['total_paid'] == 0 && $group['total_waived'] == 0) {
        $group['status'] = 'Not Paid';
    } elseif ($group['total_paid'] > 0 && $group['total_paid'] < $group['total_credit']) {
        $group['status'] = 'Partially Paid';
    } elseif ($group['total_waived'] > 0 && $group['total_paid'] == 0) {
        $group['status'] = 'Waived';
    } else {
        $group['status'] = 'Partially Paid';
    }
}
unset($group);
// Separate cleared credits (total_balance == 0)
foreach ($credits_by_customer as $cid => $group) {
    if ($group['total_balance'] == 0) {
        $cleared_credits_by_customer[$cid] = $group;
        unset($credits_by_customer[$cid]);
    }
}

// --- Fetch all credit transactions for grouping cleared credits ---
$all_credits_query = "SELECT ct.*, s.invoice_number, s.created_at as sale_date, 
    c.name as customer_name,
    GROUP_CONCAT(p.name SEPARATOR ', ') as items,
    SUM(si.quantity) as total_quantity,
    s.payment_method as sale_payment_method,
    s.paid_amount as sale_paid_amount,
    GROUP_CONCAT(DISTINCT sp.method SEPARATOR ', ') as payment_methods,
    GROUP_CONCAT(DISTINCT sp.amount SEPARATOR ', ') as payment_amounts,
    COALESCE(SUM(cp.amount), 0) as amount_paid,
    CASE 
        WHEN EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id) THEN 'paid'
        ELSE 'pending'
    END as status
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    JOIN customers c ON ct.customer_id = c.id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    LEFT JOIN sale_items si ON s.id = si.sale_id
    LEFT JOIN products p ON si.product_id = p.id
    LEFT JOIN sale_payments sp ON s.id = sp.sale_id
    GROUP BY ct.id ORDER BY s.created_at DESC";
$all_credits_stmt = $pdo->prepare($all_credits_query);
$all_credits_stmt->execute();
$all_credits = $all_credits_stmt->fetchAll();

// Group all credits by customer for cleared credits
$all_credits_by_customer = [];
$cleared_credits_by_customer = [];
foreach ($all_credits as $credit) {
    $cid = $credit['customer_id'];
    if (!isset($all_credits_by_customer[$cid])) {
        $all_credits_by_customer[$cid] = [
            'customer_name' => $credit['customer_name'] ?? $credit['name'] ?? 'Unknown',
            'total_credit' => 0,
            'total_paid' => 0,
            'total_waived' => 0,
            'total_balance' => 0,
            'transactions' => []
        ];
    }
    $paid = $credit['amount_paid'];
    // Calculate waived for this transaction
    $waiveStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
    $waiveStmt->execute([$credit['id']]);
    $waived = $waiveStmt->fetchColumn();
    $all_credits_by_customer[$cid]['total_credit'] += $credit['amount'];
    $all_credits_by_customer[$cid]['total_paid'] += $paid;
    $all_credits_by_customer[$cid]['total_waived'] += $waived;
    $all_credits_by_customer[$cid]['total_balance'] += $credit['amount'] - ($paid + $waived);
    $credit['amount_waived'] = $waived;
    $all_credits_by_customer[$cid]['transactions'][] = $credit;
}
// Separate cleared credits (total_balance == 0)
foreach ($all_credits_by_customer as $cid => $group) {
    if ($group['total_credit'] > 0 && abs($group['total_balance']) < 0.01) {
        $cleared_credits_by_customer[$cid] = $group;
    }
}

if (isset($_GET['fetch_unpaid_invoices']) && $_GET['fetch_unpaid_invoices'] == '1' && isset($_GET['customer_id'])) {
    $customerId = intval($_GET['customer_id']);
    $stmt = $pdo->prepare("
        SELECT ct.id, s.invoice_number, s.created_at as sale_date, ct.amount, 
               (SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ct.id) as paid
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        WHERE ct.customer_id = ?
        GROUP BY ct.id
        HAVING ct.amount - paid > 0
        ORDER BY s.created_at ASC
    ");
    $stmt->execute([$customerId]);
    $invoices = [];
    while ($row = $stmt->fetch()) {
        $invoices[] = [
            'id' => $row['id'],
            'invoice_number' => $row['invoice_number'],
            'sale_date' => $row['sale_date'],
            'balance' => number_format($row['amount'] - $row['paid'], 2)
        ];
    }
    if (empty($invoices)) {
        echo json_encode([
            'success' => true,
            'invoices' => [],
            'debug' => [
                'customer_id' => $customerId,
                'sql' => 'SELECT ct.id, s.invoice_number, s.created_at as sale_date, ct.amount, (SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ct.id) as paid FROM credit_transactions ct JOIN sales s ON ct.sale_id = s.id WHERE ct.customer_id = ? GROUP BY ct.id HAVING ct.amount - paid > 0 ORDER BY s.created_at ASC',
                'message' => 'No unpaid invoices found for this customer.'
            ]
        ]);
        exit;
    } else {
        echo json_encode([
            'success' => true,
            'invoices' => $invoices,
            'debug' => [
                'customer_id' => $customerId,
                'count' => count($invoices),
                'message' => 'Unpaid invoices found.'
            ]
        ]);
        exit;
    }
}

// ... existing code ...
// Handle unpaid invoices request
if (isset($_GET['get_unpaid_invoices']) && isset($_GET['customer_id'])) {
    $customerId = $_GET['customer_id'];
    
    try {
        // Get unpaid invoices for the customer
        $stmt = $pdo->prepare("
            SELECT 
                ct.id,
                s.invoice_number,
                ct.amount - COALESCE(SUM(cp.amount), 0) as balance
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            GROUP BY ct.id
            HAVING balance > 0
            ORDER BY s.created_at DESC
        ");
        $stmt->execute([$customerId]);
        $invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format the balance for display
        foreach ($invoices as &$invoice) {
            $invoice['balance'] = number_format($invoice['balance'], 2);
        }
        
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'invoices' => $invoices]);
        exit;
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Error fetching unpaid invoices: ' . $e->getMessage()]);
        exit;
    }
}

// ... existing code ...

// Fetch all products for the add-credit-item modal
$products = $pdo->query("SELECT id, name, price FROM products ORDER BY name")->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_credit_customer_id'])) {
    header('Content-Type: application/json');
    try {
        $customerId = intval($_POST['add_credit_customer_id']);
        $productId = intval($_POST['product_id']);
        $quantity = intval($_POST['quantity']);
        $price = floatval($_POST['price']);
        $userId = $_SESSION['user_id'];
        if ($customerId < 1 || $productId < 1 || $quantity < 1 || $price < 0) {
            throw new Exception('Invalid input.');
        }
        $pdo->beginTransaction();
        
        // Generate unique invoice number
        $stmt = $pdo->query("SELECT MAX(CAST(SUBSTRING(invoice_number, 4) AS UNSIGNED)) as max_num FROM sales WHERE invoice_number LIKE 'INV%'");
        $result = $stmt->fetch();
        $nextNum = ($result['max_num'] ?? 0) + 1;
        $invoiceNumber = 'INV' . str_pad($nextNum, 6, '0', STR_PAD_LEFT);
        
        // Insert sale
        $stmt = $pdo->prepare("INSERT INTO sales (invoice_number, customer_id, user_id, total_amount, paid_amount, payment_method, status, payment_status, created_at, updated_at) VALUES (?, ?, ?, ?, 0, 'credit', 'pending', 'pending', NOW(), NOW())");
        $total = $quantity * $price;
        $stmt->execute([$invoiceNumber, $customerId, $userId, $total]);
        $saleId = $pdo->lastInsertId();
        // Insert sale item
        $stmt = $pdo->prepare("INSERT INTO sale_items (sale_id, product_id, quantity, price, total, discount, unit_price, total_price) VALUES (?, ?, ?, ?, ?, 0, ?, ?)");
        $stmt->execute([$saleId, $productId, $quantity, $price, $total, $price, $total]);
        // Insert credit transaction
        $stmt = $pdo->prepare("INSERT INTO credit_transactions (customer_id, sale_id, amount, type, status, due_date, created_at) VALUES (?, ?, ?, 'credit', 'pending', DATE_ADD(NOW(), INTERVAL 30 DAY), NOW())");
        $stmt->execute([$customerId, $saleId, $total]);
        $pdo->commit();
        echo json_encode(['success' => true, 'message' => 'Credit item added successfully.']);
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Credit Sales - POS System</title>
    <?php
    // Get favicon path from database
    $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'store_favicon' AND setting_group = 'store_info' LIMIT 1");
    $stmt->execute();
    $faviconPath = $stmt->fetchColumn();
    
    if ($faviconPath && file_exists('../' . $faviconPath)) {
        echo '<link rel="icon" type="image/x-icon" href="../' . htmlspecialchars($faviconPath) . '">';
        echo '<link rel="shortcut icon" type="image/x-icon" href="../' . htmlspecialchars($faviconPath) . '">';
    } else {
        echo '<link rel="icon" type="image/x-icon" href="../assets/images/favicon.ico">';
        echo '<link rel="shortcut icon" type="image/x-icon" href="../assets/images/favicon.ico">';
    }
    ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { 
            background: #f8fafc; 
            font-family: 'Inter', sans-serif; 
        }
        .card { 
            border-radius: 16px; 
            box-shadow: 0 4px 20px rgba(0,0,0,0.05); 
            border: none; 
            transition: transform 0.2s;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .card-header { 
            background: none; 
            border-bottom: none; 
            padding: 1.5rem 1.5rem 0.5rem;
        }
        .card-body {
            padding: 1.5rem;
        }
        .stat-card {
            background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
            border-left: 4px solid;
        }
        .stat-card.primary { border-left-color: #3b82f6; }
        .stat-card.success { border-left-color: #10b981; }
        .stat-card.info { border-left-color: #06b6d4; }
        .stat-card.warning { border-left-color: #f59e0b; }
        .stat-value {
            font-size: 1.8rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0.5rem 0;
        }
        .stat-label {
            color: #64748b;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        .table {
            background: #fff;
            border-radius: 12px;
            overflow: hidden;
        }
        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: #475569;
            border-bottom: 2px solid #e2e8f0;
        }
        .table td {
            vertical-align: middle;
            color: #334155;
        }
        .btn {
            border-radius: 8px;
            font-weight: 500;
            padding: 0.5rem 1rem;
        }
        .btn-primary {
            background: #3b82f6;
            border: none;
        }
        .btn-primary:hover {
            background: #2563eb;
        }
        .filter-card {
            background: #fff;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
        }
        .filter-card .form-control {
            border-radius: 6px;
            border: 1px solid #e2e8f0;
        }
        .filter-card .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .badge {
            padding: 0.5em 1em;
            border-radius: 6px;
            font-weight: 500;
        }
        .badge-paid {
            background: #10b981;
            color: white;
        }
        .badge-pending {
            background: #f59e0b;
            color: white;
        }
        .badge-warning, .badge-partially-paid {
            background: #fbbf24;
            color: #111;
        }
        .progress {
            height: 8px;
            border-radius: 4px;
            background: #e2e8f0;
        }
        .progress-bar {
            border-radius: 4px;
        }
        /* Credit Table Column Widths */
        .credit-table th:nth-child(1), .credit-table td:nth-child(1) { /* Date */
            width: 10%;
            min-width: 80px;
        }
        .credit-table th:nth-child(2), .credit-table td:nth-child(2) { /* Invoice */
            width: 12%;
            min-width: 100px;
        }
        .credit-table th:nth-child(3), .credit-table td:nth-child(3) { /* Items */
            width: 20%;
            min-width: 150px;
        }
        .credit-table th:nth-child(4), .credit-table td:nth-child(4) { /* Qty */
            width: 6%;
            min-width: 50px;
            text-align: center;
        }
        .credit-table th:nth-child(5), .credit-table td:nth-child(5) { /* Payment Details */
            width: 18%;
            min-width: 140px;
        }
        .credit-table th:nth-child(6), .credit-table td:nth-child(6) { /* Credit Amount */
            width: 12%;
            min-width: 100px;
            text-align: right;
        }
        .credit-table th:nth-child(7), .credit-table td:nth-child(7) { /* Paid */
            width: 10%;
            min-width: 80px;
            text-align: right;
        }
        .credit-table th:nth-child(8), .credit-table td:nth-child(8) { /* Balance */
            width: 12%;
            min-width: 100px;
            text-align: right;
        }
        .credit-table th:nth-child(9), .credit-table td:nth-child(9) { /* Status */
            width: 10%;
            min-width: 120px;
            text-align: center;
        }
        /* Additional table styling */
        .credit-table {
            font-size: 0.875rem;
        }
        .credit-table th {
            font-weight: 600;
            background: #f8fafc;
            border-bottom: 2px solid #e2e8f0;
            white-space: nowrap;
        }
        .credit-table td {
            border-bottom: 1px solid #e2e8f0;
            padding: 0.5rem 0.25rem;
        }
        .credit-table td:nth-child(3) { /* Items column */
            max-width: 0;
            word-wrap: break-word;
            white-space: normal;
            overflow-wrap: break-word;
            hyphens: auto;
        }
        .credit-table td:nth-child(5) { /* Payment Details column */
            max-width: 0;
            word-wrap: break-word;
            white-space: normal;
            overflow-wrap: break-word;
            hyphens: auto;
        }
        /* Print Report Styles */
        @media print {
            body *:not(.print-section):not(.print-section *) {
                display: none !important;
            }
            .print-section {
                display: block !important;
                width: 100%;
                margin: 0;
                padding: 0;
            }
            .print-footer {
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                text-align: center;
                font-size: 0.95rem;
                color: #444;
                border-top: 1px solid #e5e7eb;
                background: #fff;
                padding: 8px 0 4px 0;
            }
            .print-footer .page-number:after {
                content: "Page " counter(page) " of " counter(pages);
            }
        }
        /* Dropdown styles */
        .dropdown-menu {
            min-width: 250px;
            padding: 0.5rem 0;
            margin: 0;
            border: 1px solid rgba(0,0,0,.15);
            border-radius: 0.25rem;
            box-shadow: 0 0.5rem 1rem rgba(0,0,0,.175);
            z-index: 9999 !important;
            position: absolute !important;
        }
        .dropdown-item {
            padding: 0.5rem 1rem;
            clear: both;
            font-weight: 400;
            color: #212529;
            text-align: inherit;
            white-space: nowrap;
            background-color: transparent;
            border: 0;
            cursor: pointer;
        }
        .dropdown-item:hover {
            color: #1e2125;
            background-color: #e9ecef;
        }
        .dropdown-item-form {
            margin: 0;
            padding: 0;
        }
        .dropdown-item-form button {
            width: 100%;
            text-align: left;
            padding: 0.5rem 1rem;
            border: none;
            background: none;
        }
        .dropdown-item-form button:hover {
            background-color: #e9ecef;
        }
        .btn-group {
            position: relative;
        }
        /* Add styles for the reminder dropdown */
        .reminder-dropdown {
            position: relative;
        }
        .reminder-dropdown .dropdown-menu {
            transform: translate3d(0px, 38px, 0px) !important;
            top: 0 !important;
            left: 0 !important;
            will-change: transform;
        }
        /* Ensure the dropdown stays above other content */
        .table-responsive {
            position: relative;
        }
        .table-responsive .dropdown-menu {
            position: absolute !important;
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <!-- Main Content -->
        <div class="col-md-10 ms-sm-auto px-md-4">
            <div class="container-fluid mt-4">
                <!-- Page Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="mb-0">Credit Sales Management</h2>
                    <a href="Cleared Credit Sales.php" target="_blank" class="btn btn-outline-success" id="showClearedCreditsBtn">
                        <i class="fas fa-check-circle"></i> Cleared Credit Sales
                    </a>
                </div>

                <!-- Statistics Cards -->
                <div class="row mb-4 no-print-stats">
                    <div class="col-md-3">
                        <div class="card stat-card primary">
                            <div class="card-body">
                                <div class="stat-label">Total Credit</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_credit, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-primary" style="width: 100%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card success">
                            <div class="card-body">
                                <div class="stat-label">Amount Paid</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_paid, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-success" style="width: <?php echo ($total_credit > 0) ? ($total_paid / $total_credit * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card warning">
                            <div class="card-body">
                                <div class="stat-label">Pending Amount</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_pending, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-warning" style="width: <?php echo ($total_credit > 0) ? ($total_pending / $total_credit * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card info">
                            <div class="card-body">
                                <div class="stat-label">Credit Customers</div>
                                <div class="stat-value"><?php echo number_format($total_customers); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-info" style="width: 100%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4 no-print-filters">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Status</label>
                                <select name="status" class="form-select">
                                    <option value="">All Status</option>
                                    <option value="paid" <?php echo $status === 'paid' ? 'selected' : ''; ?>>Paid</option>
                                    <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Customer</label>
                                <select name="customer_id" class="form-select">
                                    <option value="">All Customers</option>
                                    <?php foreach ($customers as $customer): ?>
                                    <option value="<?php echo $customer['id']; ?>" <?php echo $customer_id == $customer['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($customer['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Search</label>
                                <input type="text" name="search" class="form-control" placeholder="Search by invoice or customer..." value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <button type="submit" class="btn btn-primary w-100">Apply Filters</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Cleared Credit Sales Table (hidden by default) -->
                <div id="clearedCreditsSection" style="display:none; margin-bottom:2rem;">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">Cleared Credit Sales</h5>
                        </div>
                        <div class="card-body p-0">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0" id="clearedCreditsTable">
                                    <thead>
                                        <tr>
                                            <th>Customer</th>
                                            <th>Total Credit</th>
                                            <th>Total Paid</th>
                                            <th>Total Balance</th>
                                            <th>Credits</th>
                                            <th>Print</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($cleared_credits_by_customer as $cid => $group): ?>
                                        <tr class="cleared-customer-row" data-customer-id="<?php echo $cid; ?>">
                                            <td>
                                                <a href="#" class="toggle-cleared-transactions" data-customer-id="<?php echo $cid; ?>" style="text-decoration:underline; cursor:pointer;">
                                                    <?php echo htmlspecialchars($group['customer_name']); ?>
                                                </a>
                                            </td>
                                            <td><?php echo $currency . ' ' . number_format($group['total_credit'], 2); ?></td>
                                            <td><?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></td>
                                            <td><?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></td>
                                            <td><span class="badge bg-info"><?php echo count($group['transactions']); ?> credits</span></td>
                                            <td><button class="btn btn-sm btn-outline-primary print-cleared-receipt" data-customer-id="<?php echo $cid; ?>"><i class="fas fa-print"></i> Print</button></td>
                                        </tr>
                                        <?php foreach ($group['transactions'] as $credit): ?>
                                        <?php if ($credit['status'] !== 'paid') continue; ?>
                                        <tr class="cleared-transaction-row" data-customer-id="<?php echo $cid; ?>" style="display:none; background:#f8fafc;">
                                            <td colspan="1"></td>
                                            <td colspan="5">
                                                <div style="display:flex; flex-direction:column; gap:8px;">
                                                    <div style="display:flex; gap:24px; align-items:center;">
                                                        <span><b>Date:</b> <?php echo date('M d, Y', strtotime($credit['sale_date'])); ?></span>
                                                        <span><b>Invoice:</b> <?php echo htmlspecialchars($credit['invoice_number']); ?></span>
                                                        <span><b>Amount:</b> <?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></span>
                                                        <span><b>Paid:</b> <?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></span>
                                                        <span><b>Status:</b> <?php echo ucfirst($credit['status']); ?></span>
                                                    </div>
                                                    <div style="padding-left:16px;">
                                                        <b>Items:</b> <?php echo htmlspecialchars($credit['items'] ?? 'No items found'); ?>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                        <!-- Hidden receipt template for this customer -->
                                        <template id="receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-paid="<?php echo $group['total_paid']; ?>">
                                            <div style="width:80mm; font-family:monospace; font-size:13px; color:#222;">
                                                <div style="text-align:center; margin-bottom:8px;">
                                                    <div style="font-size:1.2rem; font-weight:700; color:#222;">POS</div>
                                                    <div><?php echo htmlspecialchars($storeAddress); ?></div>
                                                    <div>Phone: <?php echo htmlspecialchars($storePhone); ?></div>
                                                    <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                                                    <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                                                </div>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
                                                <div><b>Date:</b> <?php echo date('Y-m-d H:i:s'); ?></div>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div><b>Cleared Credit Items:</b></div>
                                                <table style="width:100%; font-size:12px; border-collapse:collapse;">
                                                    <thead>
                                                        <tr>
                                                            <th style="text-align:left;">Date</th>
                                                            <th style="text-align:left;">Invoice</th>
                                                            <th style="text-align:right;">Amount</th>
                                                            <th style="text-align:right;">Waived</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($group['transactions'] as $item): ?>
                                                        <?php if ($item['status'] !== 'paid') continue; ?>
                                                        <?php
                                                        // Calculate total waived for this credit transaction
                                                        $waived = 0;
                                                        $waiveStmt = $pdo->prepare("SELECT SUM(amount) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
                                                        $waiveStmt->execute([$item['id']]);
                                                        $waived = $waiveStmt->fetchColumn();
                                                        ?>
                                                        <tr>
                                                            <td><?php echo date('M d', strtotime($item['sale_date'])); ?></td>
                                                            <td><?php echo htmlspecialchars($item['invoice_number']); ?></td>
                                                            <td style="text-align:right;"><?php echo $currency . ' ' . number_format($item['amount'], 2); ?></td>
                                                            <td style="text-align:right; color:#dc2626;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                                                        </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div><b>Total Paid:</b> <?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></div>
                                                <div><b>Total Waived:</b> <?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></div>
                                                <div style="margin:12px 0 4px 0; text-align:center;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
                                                <div style="text-align:center; margin:8px 0;">
                                                    <img id="receipt-qr" style="width:120px; height:120px;" />
                                                </div>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div style="text-align:center; font-size:11px;">Designed and developed by Vico Technologies</div>
                                            </div>
                                        </template>
                                        <?php endforeach; ?>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Credit Transactions Table -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">Credit Transactions</h5>
                        <span class="badge bg-primary">Grouped by Customer</span>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0" id="creditTransactionsTable">
                                <thead>
                                    <tr>
                                        <th>Customer</th>
                                        <th>Total Credit</th>
                                        <th>Total Paid</th>
                                        <th>Total Waived</th>
                                        <th>Total Balance</th>
                                        <th>Status</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($credits_by_customer as $cid => $group): ?>
                                    <?php
                                    // Filter out customers with only fully paid invoices
                                    $has_unpaid = false;
                                    foreach ($group['transactions'] as $credit) {
                                        $item_balance = $credit['amount'] - $credit['amount_paid'];
                                        if ($item_balance > 0.001) {
                                            $has_unpaid = true;
                                            break;
                                        }
                                    }
                                    if (!$has_unpaid) continue;
                                    ?>
                                    <tr class="customer-row" data-customer-id="<?php echo $cid; ?>">
                                        <td>
                                            <a href="#" class="customer-name" data-customer-id="<?php echo $cid; ?>" style="text-decoration:none; color:inherit; display:flex; align-items:center; gap:8px;">
                                                <i class="fas fa-chevron-right" style="font-size:0.8rem;"></i>
                                                <?php echo htmlspecialchars($group['customer_name']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_credit'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></td>
                                        <td>
                                            <?php
                                            $statusClass = '';
                                            switch($group['status']) {
                                                case 'Not Paid':
                                                    $statusClass = 'bg-danger';
                                                    break;
                                                case 'Partially Paid':
                                                    $statusClass = 'bg-warning text-dark';
                                                    break;
                                                default:
                                                    $statusClass = 'bg-danger';
                                            }
                                            ?>
                                            <span class="badge <?php echo $statusClass; ?>"><?php echo $group['status']; ?></span>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <button type="button" class="btn btn-sm btn-success pay-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" 
                                                    data-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-money-bill"></i> Pay
                                                </button>
                                                <button type="button" class="btn btn-sm btn-warning waive-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" 
                                                    data-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-hand-holding-heart"></i> Waive
                                                </button>
                                                <button type="button" class="btn btn-sm btn-info reminder-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" 
                                                    data-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-bell"></i> Remind
                                                </button>
                                                <button type="button" class="btn btn-sm btn-primary print-credit-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>">
                                                    <i class="fas fa-print"></i> Print
                                                </button>
                                                <button type="button" class="btn btn-sm btn-dark add-credit-item-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>">
                                                    <i class="fas fa-plus"></i> Add
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    <tr class="credit-items" data-customer-id="<?php echo $cid; ?>" style="display:none; background:#f8fafc;">
                                        <td colspan="6">
                                            <div class="p-3">
                                                <h6 class="mb-3">Credit Items for <?php echo htmlspecialchars($group['customer_name']); ?></h6>
                                                <div class="table-responsive">
                                                    <table class="table table-sm credit-table">
                                                        <thead>
                                                            <tr style="background:#f1f5f9; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Date</th>
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Invoice</th>
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Items</th>
                                                                <th style="text-align:center; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Qty</th>
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Payment Details</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Waived</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Credit Amount</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Paid</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Balance</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php $rowAlt = false; foreach ($group['transactions'] as $credit): ?>
                                                            <?php
                                                            // Calculate waived amount for this transaction
                                                            $waiveStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
                                                            $waiveStmt->execute([$credit['id']]);
                                                            $waived = $waiveStmt->fetchColumn();
                                                            $item_balance = $credit['amount'] - $credit['amount_paid'] - $waived; 
                                                            if ($item_balance <= 0.001) continue; 
                                                            ?>
                                                            <tr style="background:<?php echo $rowAlt ? '#f8fafc' : '#fff'; ?>; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo date('M d', strtotime($credit['sale_date'])); ?></td>
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['items'] ?? 'No items found'); ?></td>
                                                                <td style="text-align:center; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $credit['total_quantity'] ?? 0; ?></td>
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                                                                    <?php
                                                                    $paymentDetails = '';
                                                                    if (!empty($credit['payment_methods'])) {
                                                                        $methods = explode(', ', $credit['payment_methods']);
                                                                        $amounts = explode(', ', $credit['payment_amounts']);
                                                                        $details = [];
                                                                        for ($i = 0; $i < count($methods); $i++) {
                                                                            if (isset($amounts[$i]) && $amounts[$i] > 0) {
                                                                                $details[] = ucfirst($methods[$i]) . ': ' . $currency . ' ' . number_format($amounts[$i], 2);
                                                                            }
                                                                        }
                                                                        $paymentDetails = implode(', ', $details);
                                                                    } elseif (!empty($credit['sale_payment_method']) && $credit['sale_paid_amount'] > 0) {
                                                                        $paymentDetails = ucfirst($credit['sale_payment_method']) . ': ' . $currency . ' ' . number_format($credit['sale_paid_amount'], 2);
                                                                    } else {
                                                                        $paymentDetails = 'Credit Sale';
                                                                    }
                                                                    echo $paymentDetails;
                                                                    ?>
                                                                </td>
                                                                <td style="text-align:right; padding:5px 4px; color:#dc2626; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                                                                <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></td>
                                                                <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                                                                <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($item_balance, 2); ?></td>
                                                            </tr>
                                                            <?php $rowAlt = !$rowAlt; endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>


<!-- Payment Modal -->
<div class="modal fade" id="paymentModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Process Payment</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="paymentForm" method="POST">
                    <input type="hidden" name="payment_id" id="paymentId">
                    <input type="hidden" name="payment_customer_id" id="paymentCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="customerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Unpaid Invoices</label>
                        <div id="paymentInvoiceList" style="max-height:150px; overflow-y:auto; border:1px solid #eee; border-radius:6px; padding:8px; background:#fafbfc;"></div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Amount</label>
                        <input type="number" class="form-control" name="amount" id="paymentAmount" step="0.01" min="1" required>
                        <div class="invalid-feedback">Minimum payment amount is 1</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Payment Method</label>
                        <select class="form-select" name="payment_method" id="paymentMethod" required>
                            <option value="cash">Cash</option>
                            <option value="mpesa">M-Pesa</option>
                        </select>
                    </div>
                    <div class="mb-3" id="mpesaPhoneGroup" style="display: none;">
                        <label class="form-label">M-Pesa Phone Number</label>
                        <input type="text" class="form-control" id="mpesaPhone" name="mpesa_phone" placeholder="e.g., 254712345678">
                        <div class="invalid-feedback">Please enter a valid M-Pesa phone number</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Reference (Optional)</label>
                        <input type="text" class="form-control" name="reference" placeholder="e.g., Receipt number">
            </div>
                    <div class="text-end">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Process Payment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Payment Receipt Template -->
<template id="payment-receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-balance="<?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?>">
    <div style="width:80mm; font-family: Arial, Helvetica, sans-serif; font-size:13px; color:#111; font-weight:700;">
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.1rem; font-weight:900; color:#000; margin-bottom:2px;">PAYMENT RECEIPT</div>
            <?php if ($logoData): ?>
                <?php
                $mimeType = 'image/jpeg'; // Default to JPEG
                if (strpos(strtolower($logoName), '.png') !== false) {
                    $mimeType = 'image/png';
                } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                    $mimeType = 'image/gif';
                }
                ?>
                <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
            <?php endif; ?>
        </div>
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.1rem; font-weight:700; color:#111; font-family: Arial, Helvetica, sans-serif;"><?php echo htmlspecialchars($storeName); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Phone: <?php echo htmlspecialchars($storePhone); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Email: <?php echo htmlspecialchars($storeEmail); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
        </div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Date:</b> <span class="receipt-date"></span></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Receipt No:</b> <span class="receipt-number"></span></div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-weight:700; margin-bottom:4px; font-family: Arial, Helvetica, sans-serif;">Payment Details:</div>
        <table style="width:100%; font-size:12px; border-collapse:collapse; margin-bottom:10px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <thead>
                <tr style="background:#f1f5f9; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Date</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Invoice</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Amount</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Paid</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Balance</th>
                </tr>
            </thead>
            <tbody>
                <?php $rowAlt = false; foreach ($group['transactions'] as $credit): ?>
                <?php $item_balance = $credit['amount'] - $credit['amount_paid']; if ($item_balance <= 0.001) continue; ?>
                <tr style="background:<?php echo $rowAlt ? '#f8fafc' : '#fff'; ?>; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo date('M d', strtotime($credit['sale_date'])); ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($item_balance, 2); ?></td>
                </tr>
                <?php $rowAlt = !$rowAlt; endforeach; ?>
            </tbody>
        </table>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Balance:</b> <?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></div>
        <div style="margin:14px 0 6px 0; text-align:center; font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Paid:</b> <?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Waived:</b> <?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></div>
    </div>
</template>

<!-- Add waive Modal -->
<div class="modal fade" id="waiveModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Waive Amount</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="waiveForm" method="POST">
                    <input type="hidden" name="waive_customer_id" id="waiveCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="waiveCustomerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Unpaid Invoices</label>
                        <div id="waiveInvoiceList" style="max-height:150px; overflow-y:auto; border:1px solid #eee; border-radius:6px; padding:8px; background:#fafbfc;"></div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Amount to waive</label>
                        <input type="number" class="form-control" name="waive_amount" id="waiveAmount" step="0.01" min="1" required>
                        <div class="invalid-feedback">Minimum waive amount is 1</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Reason</label>
                        <textarea class="form-control" name="waive_reason" id="waiveReason" rows="3" required></textarea>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-warning">Waive Amount</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Add waive Receipt Template -->
<template id="waive-receipt-template">
    <div style="width:80mm; font-family: 'Sans I Am', Arial, sans-serif; font-size:13px; color:#111; font-weight:600; padding:0; margin:0;">
        <div style="width:100%; text-align:center; color:#000; font-weight:900; margin-bottom:2px; font-size:1.1rem;">WAIVER RECEIPT</div>
        <?php if ($logoData): ?>
            <?php
            $mimeType = 'image/jpeg'; // Default to JPEG
            if (strpos(strtolower($logoName), '.png') !== false) {
                $mimeType = 'image/png';
            } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                $mimeType = 'image/gif';
            }
            ?>
            <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
        <?php endif; ?>
        <div style="text-align:center; margin-bottom:8px; width:100%;">
            <div style="font-size:1.2rem; font-weight:700; color:#000;"><?php echo htmlspecialchars($storeName); ?></div>
            <div style="width:100%; color:#111; font-weight:600; "><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="width:100%; color:#111; font-weight:600;">Phone: <?php echo htmlspecialchars($storePhone); ?></div>
            <div style="width:100%; color:#111; font-weight:600;">Email: <?php echo htmlspecialchars($storeEmail); ?></div>
            <div style="width:100%; color:#111; font-weight:600;">Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
        </div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="width:100%; color:#111;"><b>Date:</b> <span id="waive-receipt-date"></span></div>
        <div style="width:100%; color:#111;"><b>Time:</b> <span id="waive-receipt-time"></span></div>
        <div style="width:100%; color:#111;"><b>Customer:</b> <span id="waive-receipt-customer"></span></div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="width:100%; color:#000; font-weight:700; text-align:center;">Waived Invoices</div>
        <div id="waive-receipt-waive-details"></div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="width:100%; color:#000;"><b>Previous Balance:</b> <span id="waive-receipt-prev-balance"></span></div>
        <div style="width:100%; color:#000;"><b>Amount Waived:</b> <span id="waive-receipt-amount"></span></div>
        <div style="width:100%; color:#000;"><b>Reason:</b> <span id="waive-receipt-reason"></span></div>
        <div style="width:100%; color:#000;"><b>Remaining Amount:</b> <span id="waive-receipt-balance"></span></div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="margin:12px 0 4px 0; text-align:center; width:100%; color:#000; font-weight:700;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
        <div style="text-align:center; margin:8px 0; width:100%;">
            <img id="waive-receipt-qr" style="width:120px; height:120px;" />
        </div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="text-align:center; font-size:11px; width:100%; color:#111; font-weight:600;">Designed and developed by Vicotechnologies.co.ke<br>+254702214793</div>
    </div>
</template>

<script>
function showPaymentModal(creditId, amount, customerName) {
    document.getElementById('paymentId').value = creditId;
    document.getElementById('paymentAmount').value = amount;
    document.getElementById('customerName').value = customerName;
    
    // Fetch unpaid invoices
    fetch(`credit.php?get_unpaid_invoices=1&customer_id=${creditId}`)
        .then(response => response.json())
        .then(data => {
            const invoiceList = document.getElementById('paymentInvoiceList');
            invoiceList.innerHTML = '';
            
            if (data.invoices && data.invoices.length > 0) {
                data.invoices.forEach(invoice => {
                    const div = document.createElement('div');
                    div.className = 'form-check mb-2';
                    div.innerHTML = `
                        <input class="form-check-input" type="checkbox" name="payment_invoice_ids[]" value="${invoice.id}" id="invoice_${invoice.id}">
                        <label class="form-check-label" for="invoice_${invoice.id}">
                            Invoice #${invoice.invoice_number} - Balance: ${invoice.balance}
                        </label>
                    `;
                    invoiceList.appendChild(div);
                });
            } else {
                invoiceList.innerHTML = '<div class="text-muted">No unpaid invoices found</div>';
            }
        })
        .catch(error => {
            console.error('Error fetching unpaid invoices:', error);
            document.getElementById('paymentInvoiceList').innerHTML = '<div class="text-danger">Error loading unpaid invoices</div>';
        });
    
    const modal = new bootstrap.Modal(document.getElementById('paymentModal'));
    modal.show();
}

document.getElementById('paymentMethod').addEventListener('change', function() {
    const mpesaFields = document.getElementById('mpesaFields');
    mpesaFields.style.display = this.value === 'mpesa' ? 'block' : 'none';
});

function printReceipt(saleId) {
    // Open receipt in a new window
    const receiptWindow = window.open(`print_receipt.php?sale_id=${saleId}`, '_blank', 'width=800,height=600');
    
    // Wait for the window to load
    receiptWindow.onload = function() {
        // Print the receipt
        receiptWindow.print();
        
        // Close the window after printing (optional)
        // receiptWindow.close();
    };
}

function printCreditReport() {
    window.print();
}

document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.toggle-transactions').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var cid = this.getAttribute('data-customer-id');
            document.querySelectorAll('.transaction-row[data-customer-id="' + cid + '"]').forEach(function(row) {
                row.style.display = (row.style.display === 'none' || row.style.display === '') ? '' : 'none';
            });
        });
    });
    // Pay button logic
    document.querySelectorAll('.pay-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var cname = this.getAttribute('data-customer-name');
            var balance = parseFloat(this.getAttribute('data-balance')) || 0;
            
            // Set the customer ID and name
            document.getElementById('paymentCustomerId').value = cid;
            document.getElementById('customerName').value = cname;
            
            // Set the maximum payment amount to the balance
            var amountInput = document.getElementById('paymentAmount');
            amountInput.value = balance.toFixed(2);
            amountInput.max = balance;
            
            // Clear previous invoice list
            document.getElementById('paymentInvoiceList').innerHTML = '<div class="text-muted">Loading invoices...</div>';
            
            // Fetch unpaid invoices for this customer
            fetch('credit.php?fetch_unpaid_invoices=1&customer_id=' + encodeURIComponent(cid))
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    var listDiv = document.getElementById('paymentInvoiceList');
                    if (data.success && Array.isArray(data.invoices) && data.invoices.length > 0) {
                        var html = '';
                        data.invoices.forEach(function(inv) {
                            html += '<div class="form-check mb-2">';
                            html += '<input class="form-check-input" type="checkbox" name="payment_invoice_ids[]" value="' + inv.id + '" checked>';
                            html += '<label class="form-check-label">';
                            html += 'Invoice: ' + inv.invoice_number + ' | Date: ' + inv.sale_date + ' | Balance: ' + inv.balance;
                            html += '</label>';
                            html += '</div>';
                        });
                        listDiv.innerHTML = html;
                    } else {
                        listDiv.innerHTML = '<div class="text-danger">No unpaid invoices found.</div>';
                    }
                })
                .catch(error => {
                    console.error('Error fetching invoices:', error);
                    document.getElementById('paymentInvoiceList').innerHTML = 
                        '<div class="text-danger">Error loading unpaid invoices. Please try again.</div>';
                });

            // Show the payment modal
            var modal = new bootstrap.Modal(document.getElementById('paymentModal'));
            modal.show();
        });
    });

    // Update payment amount when invoices are selected/deselected
    document.getElementById('paymentInvoiceList').addEventListener('change', function(e) {
        if (e.target.type === 'checkbox') {
            var total = 0;
            var checkedBoxes = this.querySelectorAll('input[type="checkbox"]:checked');
            if (checkedBoxes.length === 0) {
                document.getElementById('paymentAmount').value = '';
                return;
            }
            checkedBoxes.forEach(function(checkbox) {
                var label = checkbox.nextElementSibling;
                var balanceText = label.textContent.split('Balance: ')[1];
                if (balanceText) {
                    total += parseFloat(balanceText.replace(/[^0-9.-]+/g, ''));
                }
            });
            document.getElementById('paymentAmount').value = total.toFixed(2);
        }
    });

    // Handle payment method change
    document.getElementById('paymentMethod').addEventListener('change', function() {
        var mpesaPhoneGroup = document.getElementById('mpesaPhoneGroup');
        mpesaPhoneGroup.style.display = this.value === 'mpesa' ? 'block' : 'none';
    });

    // Receipt button logic
    document.querySelectorAll('.receipt-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var template = document.getElementById('pending-receipt-template-' + cid);
            if (template) {
                var win = window.open('', '', 'width=400,height=600');
                win.document.write('<html><head><title>Pending Credit Receipt</title></head><body style="margin:0;">');
                win.document.write(template.innerHTML);
                win.document.write('</body></html>');
                win.document.close();
                
                // Fill in date and receipt number
                var now = new Date();
                var dateStr = now.getFullYear() + '-' + String(now.getMonth()+1).padStart(2,'0') + '-' + String(now.getDate()).padStart(2,'0') + ' ' + String(now.getHours()).padStart(2,'0') + ':' + String(now.getMinutes()).padStart(2,'0') + ':' + String(now.getSeconds()).padStart(2,'0');
                var receiptNo = 'CR-' + cid + '-' + now.getTime();
                win.document.querySelector('.receipt-date').textContent = dateStr;
                win.document.querySelector('.receipt-number').textContent = receiptNo;
                
                // Gather receipt info for QR
                var receiptInfo = {
                    store: '<?php echo addslashes($storeName); ?>',
                    address: '<?php echo addslashes($storeAddress); ?>',
                    phone: '<?php echo addslashes($storePhone); ?>',
                    email: '<?php echo addslashes($storeEmail); ?>',
                    website: '<?php echo addslashes($storeWebsite); ?>',
                    date: dateStr,
                    receipt_no: receiptNo,
                    customer: <?php echo json_encode($group['customer_name']); ?>,
                    total_pending: '<?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?>'
                };
                
                // Add items
                receiptInfo.items = [];
                var rows = win.document.querySelectorAll('table tbody tr');
                rows.forEach(function(row) {
                    var cells = row.querySelectorAll('td');
                    if (cells.length === 4) {
                        receiptInfo.items.push({
                            date: cells[0].textContent,
                            invoice: cells[1].textContent,
                            amount: cells[2].textContent,
                            items: cells[3].textContent
                        });
                    }
                });
                
                // Generate QR code using qrserver
                var qrData = JSON.stringify(receiptInfo);
                var qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(qrData);
                var qrImg = win.document.getElementById('receipt-qr');
                
                // Wait for QR code to load before printing
                qrImg.onload = function() {
                    setTimeout(function() {
                        win.print();
                        setTimeout(function() {
                            win.close();
                        }, 1000);
                    }, 500);
                };
                
                // Handle QR code load error
                qrImg.onerror = function() {
                    console.error('Failed to load QR code');
                    setTimeout(function() {
                        win.print();
                        setTimeout(function() {
                            win.close();
                        }, 1000);
                    }, 500);
                };
                
                // Set QR code source
                qrImg.src = qrUrl;
            }
        });
    });
    document.getElementById('showClearedCreditsBtn').addEventListener('click', function() {
        var section = document.getElementById('clearedCreditsSection');
        section.style.display = (section.style.display === 'none' || section.style.display === '') ? 'block' : 'none';
    });
    document.querySelectorAll('.toggle-cleared-transactions').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var cid = this.getAttribute('data-customer-id');
            document.querySelectorAll('.cleared-transaction-row[data-customer-id="' + cid + '"]').forEach(function(row) {
                row.style.display = (row.style.display === 'none' || row.style.display === '') ? '' : 'none';
            });
        });
    });
    document.querySelectorAll('.print-cleared-receipt').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var template = document.getElementById('receipt-template-' + cid);
            if (template) {
                var win = window.open('', '', 'width=400,height=600');
                win.document.write('<html><head><title>Receipt</title></head><body style="margin:0;">');
                win.document.write(template.innerHTML);
                win.document.write('</body></html>');
                win.document.close();
                
                // Generate QR code data for cleared credits
                var qrData = JSON.stringify({
                    store: '<?php echo htmlspecialchars($storeName); ?>',
                    customer: template.getAttribute('data-customer-name'),
                    total_paid: template.getAttribute('data-total-paid'),
                    date: new Date().toLocaleDateString(),
                    time: new Date().toLocaleTimeString(),
                    status: 'Cleared'
                });
                
                // Generate QR code using QRServer
                var qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(qrData);
                win.document.getElementById('receipt-qr').src = qrUrl;
                
                win.focus();
                setTimeout(function() { win.print(); }, 300);
            }
        });
    });
    // Toggle credit items when clicking customer name
    document.querySelectorAll('.customer-name').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var customerId = this.getAttribute('data-customer-id');
            var itemsRow = document.querySelector(`.credit-items[data-customer-id="${customerId}"]`);
            
            if (itemsRow) {
                if (itemsRow.style.display === 'none' || itemsRow.style.display === '') {
                    itemsRow.style.display = 'table-row';
                    this.querySelector('i').classList.remove('fa-chevron-right');
                    this.querySelector('i').classList.add('fa-chevron-down');
                } else {
                    itemsRow.style.display = 'none';
                    this.querySelector('i').classList.remove('fa-chevron-down');
                    this.querySelector('i').classList.add('fa-chevron-right');
                }
            }
        });
    });
});

// Update the payment form submission handler
document.getElementById('paymentForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Get form data
    var formData = new FormData(this);
    var amount = parseFloat(formData.get('amount'));
    var customerId = formData.get('payment_customer_id');
    var customerName = document.getElementById('customerName').value;
    var paymentMethod = formData.get('payment_method');
    
    // Validate amount
    if (amount < 1) {
        document.getElementById('paymentAmount').classList.add('is-invalid');
        return;
    }

    // Show loading state
    var submitBtn = this.querySelector('button[type="submit"]');
    var originalBtnText = submitBtn.innerHTML;
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

    // Get selected invoice IDs
    var selectedInvoices = [];
    document.querySelectorAll('#paymentInvoiceList input[type="checkbox"]:checked').forEach(function(checkbox) {
        selectedInvoices.push(checkbox.value);
    });

    if (selectedInvoices.length === 0) {
        alert('Please select at least one invoice to pay');
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalBtnText;
        return;
    }

    // Add selected invoices to form data
    formData.delete('payment_invoice_ids[]'); // Remove any existing values
    selectedInvoices.forEach(function(id) {
        formData.append('payment_invoice_ids[]', id);
    });

    // Handle M-Pesa payment
    if (paymentMethod === 'mpesa') {
        var mpesaPhone = document.getElementById('mpesaPhone').value;
        if (!mpesaPhone) {
            alert('Please enter M-Pesa phone number');
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            return;
        }
        
        // Validate phone number
        if (!/^(?:254|\+254|0)?([7-9]{1}[0-9]{8})$/.test(mpesaPhone)) {
            alert('Please enter a valid M-Pesa phone number');
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            return;
        }

        // Format phone number
        mpesaPhone = mpesaPhone.replace(/^(?:254|\+254|0)/, '254');

        // Call M-Pesa API
        fetch('/api/mpesa_stkpush.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
                phone: mpesaPhone, 
                amount: amount,
                customer_id: customerId,
                customer_name: customerName
            })
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                alert('M-Pesa payment initiated. Please check your phone to complete the payment.');
                submitPayment();
            } else {
                throw new Error(data.message || 'Failed to initiate M-Pesa payment');
            }
        })
        .catch(error => {
            console.error('M-Pesa Error:', error);
            alert('Failed to initiate M-Pesa payment: ' + error.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    } else {
        // Process regular payment
        submitPayment();
    }

    function submitPayment() {
        // Log the request data for debugging
        console.log('Submitting payment with data:', {
            amount: amount,
            customerId: customerId,
            paymentMethod: paymentMethod,
            selectedInvoices: selectedInvoices
        });

        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text().then(text => {
                try {
                    const data = JSON.parse(text);
                    console.log('Server response:', data); // Debug log
                    return data;
                } catch (e) {
                    console.error('Failed to parse response:', text);
                    throw new Error('Invalid response from server');
                }
            });
        })
        .then(data => {
            if (!data || typeof data !== 'object') {
                throw new Error('Invalid response data');
            }

            if (!data.success) {
                throw new Error(data.message || 'Payment processing failed');
            }

            // Validate required data fields
            if (!data.customer || !data.amount || !data.payment_method) {
                throw new Error('Missing required payment data');
            }

            // Close the modal
            var modal = bootstrap.Modal.getInstance(document.getElementById('paymentModal'));
            modal.hide();

            // Show success message
            alert(data.message);

            // Generate and print receipt
            var receiptWindow = window.open('', '_blank', 'width=400,height=600');
            
            // Prepare receipt data with safe defaults
            const receiptData = {
                customer: data.customer || 'N/A',
                amount: parseFloat(data.amount) || 0,
                payment_method: data.payment_method || 'N/A',
                payment_details: Array.isArray(data.payment_details) ? data.payment_details : [],
                remaining_balance: parseFloat(data.remaining_balance) || 0,
                previous_balance: parseFloat(data.previous_balance) || 0
            };

            // Write receipt content
            receiptWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Payment Receipt</title>
                    <meta charset="UTF-8">
                    <style>
                        @page {
                            size: 80mm auto;
                            margin: 0;
                        }
                        body { 
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            color: #000;
                            margin: 0;
                            padding: 5px;
                            width: 80mm;
                            max-width: 80mm;
                        }
                        .header { 
                            text-align: center;
                            margin-bottom: 5px;
                            font-size: 11px;
                        }
                        .header .store-name {
                            font-size: 13px;
                            font-weight: bold;
                            margin-bottom: 2px;
                        }
                        .divider { 
                            border-top: 1px dashed #000;
                            margin: 5px 0;
                        }
                        .payment-details { 
                            margin: 5px 0;
                            font-size: 11px;
                        }
                        table {
                            width: 100%;
                            border-collapse: collapse;
                        }
                        th, td {
                            padding: 2px;
                            text-align: left;
                        }
                        th {
                            border-bottom: 1px solid #000;
                        }
                        .thank-you { 
                            text-align: center;
                            margin: 8px 0;
                            font-weight: bold;
                        }
                        .qr-code { 
                            text-align: center;
                            margin: 8px 0;
                        }
                        .qr-code img {
                            width: 100px;
                            height: 100px;
                        }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
                        <div><?php echo htmlspecialchars($storeAddress); ?></div>
                        <div>Tel: <?php echo htmlspecialchars($storePhone); ?></div>
                        <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                        <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div>Date: ${new Date().toLocaleDateString()}</div>
                        <div>Time: ${new Date().toLocaleTimeString()}</div>
                        <div>Customer: ${receiptData.customer}</div>
                        <div>Payment Method: ${receiptData.payment_method}</div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="font-weight: bold; margin-bottom: 5px;">Payment Details:</div>
                        <table>
                            <thead>
                                <tr>
                                    <th>Invoice</th>
                                    <th>Items</th>
                                    <th style="text-align: right;">Amount</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${receiptData.payment_details.map(detail => `
                                    <tr>
                                        <td>${detail.invoice_number || 'N/A'}</td>
                                        <td>${detail.items || '-'}</td>
                                        <td style="text-align: right;">${(parseFloat(detail.paid_amount) || 0).toFixed(2)}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="display: flex; justify-content: space-between;">
                            <span>Previous Balance:</span>
                            <span>${receiptData.previous_balance.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Amount Paid:</span>
                            <span>${receiptData.amount.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Remaining Balance:</span>
                            <span>${receiptData.remaining_balance.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Payment Method:</span>
                            <span>${receiptData.payment_method}</span>
                        </div>
                    </div>
                    <div class="divider"></div>
                    <div class="thank-you">
                        Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!
                    </div>
                    <div class="qr-code">
                        <img id="receipt-qr" src="" alt="Payment QR Code">
                    </div>
                    <div class="divider"></div>
                    <div style="text-align: center; font-size: 10px;">
                        Designed and developed by Vico Technologies
                    </div>
                </body>
                </html>
            `);
            receiptWindow.document.close();

            // Generate QR code data
            const qrData = JSON.stringify({
                store: '<?php echo addslashes($storeName); ?>',
                customer: receiptData.customer,
                amount: receiptData.amount,
                previous_balance: receiptData.previous_balance,
                remaining_balance: receiptData.remaining_balance,
                date: new Date().toLocaleDateString(),
                time: new Date().toLocaleTimeString(),
                status: 'Paid',
                payment_details: receiptData.payment_details
            });

            // Set QR code image source
            const qrImg = receiptWindow.document.getElementById('receipt-qr');
            qrImg.src = `https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=${encodeURIComponent(qrData)}`;

            // Wait for QR code to load before printing
            qrImg.onload = function() {
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };

            // Fallback in case QR code fails to load
            qrImg.onerror = function() {
                console.error('Failed to load QR code');
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };
        })
        .catch(error => {
            console.error('Payment Error:', error);
            alert('An error occurred while processing your request: ' + error.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    }
});

// ... existing code ...
    // Waive button logic
    document.querySelectorAll('.waive-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var cname = this.getAttribute('data-customer-name');
            var balance = parseFloat(this.getAttribute('data-balance')) || 0;
            
            // Set the customer ID and name
            document.getElementById('waiveCustomerId').value = cid;
            document.getElementById('waiveCustomerName').value = cname;
            
            // Set the maximum waive amount to the balance
            var amountInput = document.getElementById('waiveAmount');
            amountInput.value = balance.toFixed(2);
            amountInput.max = balance;
            
            // Clear previous invoice list
            document.getElementById('waiveInvoiceList').innerHTML = '<div class="text-muted">Loading invoices...</div>';
            
            // Fetch unpaid invoices for this customer
            fetch('credit.php?fetch_unpaid_invoices=1&customer_id=' + encodeURIComponent(cid))
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    var listDiv = document.getElementById('waiveInvoiceList');
                    if (data.success && Array.isArray(data.invoices) && data.invoices.length > 0) {
                        var html = '';
                        data.invoices.forEach(function(inv) {
                            html += '<div class="form-check mb-2">';
                            html += '<input class="form-check-input" type="checkbox" name="waive_invoice_ids[]" value="' + inv.id + '" checked>';
                            html += '<label class="form-check-label">';
                            html += 'Invoice: ' + inv.invoice_number + ' | Date: ' + inv.sale_date + ' | Balance: ' + inv.balance;
                            html += '</label>';
                            html += '</div>';
                        });
                        listDiv.innerHTML = html;
                    } else {
                        listDiv.innerHTML = '<div class="text-danger">No unpaid invoices found.</div>';
                    }
                })
                .catch(error => {
                    console.error('Error fetching invoices:', error);
                    document.getElementById('waiveInvoiceList').innerHTML = 
                        '<div class="text-danger">Error loading unpaid invoices. Please try again.</div>';
                });

            // Show the waive modal
            var modal = new bootstrap.Modal(document.getElementById('waiveModal'));
            modal.show();
        });
    });

    // Handle waive form submission
    document.getElementById('waiveForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Get form data
        var formData = new FormData(this);
        var amount = parseFloat(formData.get('waive_amount'));
        var customerId = formData.get('waive_customer_id');
        var reason = formData.get('waive_reason');
        
        // Validate amount
        if (amount < 1) {
            document.getElementById('waiveAmount').classList.add('is-invalid');
            return;
        }

        // Show loading state
        var submitBtn = this.querySelector('button[type="submit"]');
        var originalBtnText = submitBtn.innerHTML;
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

        // Get selected invoice IDs
        var selectedInvoices = [];
        document.querySelectorAll('#waiveInvoiceList input[type="checkbox"]:checked').forEach(function(checkbox) {
            selectedInvoices.push(checkbox.value);
        });

        if (selectedInvoices.length === 0) {
            alert('Please select at least one invoice to waive');
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            return;
        }

        // Add selected invoices to form data
        formData.delete('waive_invoice_ids[]'); // Remove any existing values
        selectedInvoices.forEach(function(id) {
            formData.append('waive_invoice_ids[]', id);
        });

        // Log the form data for debugging
        console.log('Submitting waive request with data:', {
            amount: amount,
            customerId: customerId,
            reason: reason,
            selectedInvoices: selectedInvoices
        });

        // Submit the form
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            console.log('Server response status:', response.status);
            return response.text().then(text => {
                console.log('Raw server response:', text);
                try {
                    // Try to parse the response as JSON
                    const data = JSON.parse(text);
                    if (!data || typeof data !== 'object') {
                        throw new Error('Invalid response format');
                    }
                    return data;
                } catch (e) {
                    console.error('Failed to parse response:', e);
                    // If the response is not JSON, check if it contains an error message
                    if (text.includes('error') || text.includes('Error')) {
                        throw new Error(text);
                    }
                    throw new Error('Server returned an invalid response. Please try again.');
                }
            });
        })
        .then(data => {
            if (!data.success) {
                throw new Error(data.message || 'Waive processing failed');
            }

            // Close the modal
            var modal = bootstrap.Modal.getInstance(document.getElementById('waiveModal'));
            modal.hide();

            // Show success message
            alert(data.message || 'Amount waived successfully');

            // Generate and print receipt
            var receiptWindow = window.open('', '_blank', 'width=400,height=600');
            
            // Prepare receipt data
            const receiptData = {
                customer: document.getElementById('waiveCustomerName').value,
                amount: amount,
                reason: reason,
                previous_balance: data.old_balance,
                waived_amount: data.waived_amount,
                remaining_balance: data.remaining_balance,
                waive_details: data.waive_details || []
            };

            // Write receipt content
            receiptWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Waiver Receipt</title>
                    <meta charset="UTF-8">
                    <style>
                        @page {
                            size: 80mm auto;
                            margin: 0;
                        }
                        body { 
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            color: #000;
                            margin: 0;
                            padding: 5px;
                            width: 80mm;
                            max-width: 80mm;
                        }
                        .header { 
                            text-align: center;
                            margin-bottom: 5px;
                            font-size: 11px;
                        }
                        .header .store-name {
                            font-size: 13px;
                            font-weight: bold;
                            margin-bottom: 2px;
                        }
                        .divider { 
                            border-top: 1px dashed #000;
                            margin: 5px 0;
                        }
                        .payment-details { 
                            margin: 5px 0;
                            font-size: 11px;
                        }
                        table {
                            width: 100%;
                            border-collapse: collapse;
                        }
                        th, td {
                            padding: 2px;
                            text-align: left;
                        }
                        th {
                            border-bottom: 1px solid #000;
                        }
                        .thank-you { 
                            text-align: center;
                            margin: 8px 0;
                            font-weight: bold;
                        }
                        .qr-code { 
                            text-align: center;
                            margin: 8px 0;
                        }
                        .qr-code img {
                            width: 100px;
                            height: 100px;
                        }
                    </style>
                </head>
                <body>
                    <div style="text-align:center; font-weight:900; margin-bottom:2px; font-size:1.1rem;">WAIVER RECEIPT</div>
                    <?php if ($logoData): ?>
                        <?php
                        $mimeType = 'image/jpeg'; // Default to JPEG
                        if (strpos(strtolower($logoName), '.png') !== false) {
                            $mimeType = 'image/png';
                        } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                            $mimeType = 'image/gif';
                        }
                        ?>
                        <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
                    <?php endif; ?>
                    <div class="header">
                        <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
                        <div><?php echo htmlspecialchars($storeAddress); ?></div>
                        <div>Tel: <?php echo htmlspecialchars($storePhone); ?></div>
                        <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                        <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div>Date: ${new Date().toLocaleDateString()}</div>
                        <div>Time: ${new Date().toLocaleTimeString()}</div>
                        <div>Customer: ${receiptData.customer}</div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="font-weight: bold; margin-bottom: 5px;">Waiver Details:</div>
                        <table>
                            <thead>
                                <tr>
                                    <th>Invoice</th>
                                    <th>Items</th>
                                    <th style="text-align: right;">Amount</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${receiptData.waive_details.map(detail => `
                                    <tr>
                                        <td>${detail.invoice_number || 'N/A'}</td>
                                        <td>${detail.items || '-'}</td>
                                        <td style="text-align: right;">${(parseFloat(detail.waived_amount) || 0).toFixed(2)}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="display: flex; justify-content: space-between;">
                            <span>Previous Balance:</span>
                            <span>${receiptData.previous_balance.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Amount Waived:</span>
                            <span>${receiptData.waived_amount.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Remaining Balance:</span>
                            <span>${receiptData.remaining_balance.toFixed(2)}</span>
                        </div>
                        <div style="margin-top: 5px;">
                            <div><b>Reason for Waiver:</b></div>
                            <div>${receiptData.reason}</div>
                        </div>
                    </div>
                    <div class="divider"></div>
                    <div class="thank-you">
                        Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!
                    </div>
                    <div class="qr-code">
                        <img id="receipt-qr" src="" alt="Waiver QR Code">
                    </div>
                    <div class="divider"></div>
                    <div style="text-align: center; font-size: 10px;">
                        Designed and developed by Vicotechnologies.co.ke<br>+254702214793
                    </div>
                </body>
                </html>
            `);
            receiptWindow.document.close();

            // Generate QR code data
            const qrData = JSON.stringify({
                store: '<?php echo addslashes($storeName); ?>',
                customer: receiptData.customer,
                previous_balance: receiptData.previous_balance,
                waived_amount: receiptData.waived_amount,
                remaining_balance: receiptData.remaining_balance,
                reason: receiptData.reason,
                date: new Date().toLocaleDateString(),
                time: new Date().toLocaleTimeString(),
                status: 'Waived',
                waive_details: receiptData.waive_details
            });

            // Set QR code image source
            const qrImg = receiptWindow.document.getElementById('receipt-qr');
            qrImg.src = `https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=${encodeURIComponent(qrData)}`;

            // Wait for QR code to load before printing
            qrImg.onload = function() {
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };

            // Fallback in case QR code fails to load
            qrImg.onerror = function() {
                console.error('Failed to load QR code');
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };
        })
        .catch(error => {
            console.error('Waive Error:', error);
            alert('An error occurred while processing your request: ' + error.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    });
// ... existing code ...
</script>

<!-- Show waive debug info if set -->
<?php if (isset($_SESSION['waive_debug'])): ?>
<div style="background:#fff3cd; color:#856404; border:1px solid #ffeeba; padding:12px; margin:16px 0; border-radius:6px; font-size:1.1rem;">
    <b>Waive Debug Info:</b><br>
    Old Balance: <?php echo number_format($_SESSION['waive_debug']['old_balance'], 2); ?><br>
    Waived Amount: <?php echo number_format($_SESSION['waive_debug']['waived_amount'], 2); ?><br>
    New Balance: <?php echo number_format($_SESSION['waive_debug']['new_balance'], 2); ?><br>
    <?php if (!empty($_SESSION['waive_debug']['waive_debug_details'])): ?>
    <b>Details:</b><ul>
        <?php foreach ($_SESSION['waive_debug']['waive_debug_details'] as $detail): ?>
        <li><?php echo htmlspecialchars($detail); ?></li>
        <?php endforeach; ?>
    </ul>
    <?php endif; ?>
</div>
<?php unset($_SESSION['waive_debug']); endif; ?>

<!-- Show pay debug info if set -->
<?php if (isset($_SESSION['pay_debug'])): ?>
<div style="background:#e0f7fa; color:#006064; border:1px solid #b2ebf2; padding:12px; margin:16px 0; border-radius:6px; font-size:1.1rem;">
    <b>Pay Debug Info:</b><br>
    Old Balance: <?php echo number_format($_SESSION['pay_debug']['old_balance'], 2) . '<br>'; ?>
    Paid Amount: <?php echo number_format($_SESSION['pay_debug']['paid_amount'], 2) . '<br>'; ?>
    New Balance: <?php echo number_format($_SESSION['pay_debug']['new_balance'], 2) . '<br>'; ?>
    <?php if (!empty($_SESSION['pay_debug']['pay_debug_details'])): ?>
    <b>Details:</b><ul>
        <?php foreach ($_SESSION['pay_debug']['pay_debug_details'] as $detail): ?>
        <li><?php echo htmlspecialchars($detail); ?></li>
        <?php endforeach; ?>
    </ul>
    <?php endif; ?>
</div>
<?php unset($_SESSION['pay_debug']); endif; ?>

<!-- Add Reminder Modal -->
<div class="modal fade" id="reminderModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Send Reminder</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="reminderForm" method="POST">
                    <input type="hidden" name="reminder_customer_id" id="reminderCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="reminderCustomerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Reminder Type</label>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="reminder_type[]" value="sms" id="smsReminder" checked>
                            <label class="form-check-label" for="smsReminder">
                                SMS Reminder
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="reminder_type[]" value="email" id="emailReminder" checked>
                            <label class="form-check-label" for="emailReminder">
                                Email Reminder
                            </label>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Message</label>
                        <textarea class="form-control" name="reminder_message" id="reminderMessage" rows="3" required></textarea>
                        <small class="text-muted">You can use {customer_name}, {balance}, {shop_name}, and {shop_address} as placeholders.</small>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="submit" form="reminderForm" class="btn btn-primary">Send Reminder</button>
            </div>
        </div>
    </div>
</div>

<script>
// ... existing code ...

// Add reminder button click handler
document.querySelectorAll('.reminder-btn').forEach(function(btn) {
    btn.addEventListener('click', function() {
        var cid = this.getAttribute('data-customer-id');
        var cname = this.getAttribute('data-customer-name');
        var balance = this.getAttribute('data-balance');
        var shopName = '<?php echo addslashes($storeName); ?>';
        var shopAddress = '<?php echo addslashes($storeAddress); ?>';
        var currency = '<?php echo $currency; ?>';

        document.getElementById('reminderCustomerId').value = cid;
        document.getElementById('reminderCustomerName').value = cname;

        // Set default message with actual info
        var defaultMessage = `Dear {customer_name}, this is a reminder that you have a pending payment of {balance} at {shop_name}, located at {shop_address}. Please visit our shop to make the payment.`;
        defaultMessage = defaultMessage
            .replace(/{customer_name}/g, cname)
            .replace(/{balance}/g, currency + ' ' + balance)
            .replace(/{shop_name}/g, shopName)
            .replace(/{shop_address}/g, shopAddress);
        document.getElementById('reminderMessage').value = defaultMessage;

        var modal = new bootstrap.Modal(document.getElementById('reminderModal'));
        modal.show();
    });
});

// Handle reminder form submission
document.getElementById('reminderForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Get the submit button from the modal footer
    var submitBtn = document.querySelector('#reminderModal .modal-footer button[type="submit"]');
    if (!submitBtn || submitBtn.disabled) return;
    
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';

    var formData = new FormData(this);
    var message = formData.get('reminder_message'); // Already filled with real info

    // Send reminders based on selected types
    var reminderTypes = formData.getAll('reminder_type[]');
    var promises = [];

    reminderTypes.forEach(function(type) {
        if (type === 'sms') {
            promises.push(
                fetch('../helpers/send_reminder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        type: 'sms',
                        customer_id: formData.get('reminder_customer_id'),
                        message: message
                    })
                }).then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
            );
        } else if (type === 'email') {
            promises.push(
                fetch('../helpers/send_reminder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        type: 'email',
                        customer_id: formData.get('reminder_customer_id'),
                        message: message
                    })
                }).then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
            );
        }
    });

    Promise.all(promises)
        .then(results => {
            var success = results.every(r => r.success);
            var message = '';
            
            if (success) {
                message = 'Reminders sent successfully!';
                var modal = bootstrap.Modal.getInstance(document.getElementById('reminderModal'));
                modal.hide();
            } else {
                var failedTypes = [];
                results.forEach((result, index) => {
                    if (!result.success) {
                        failedTypes.push(reminderTypes[index]);
                    }
                });
                message = 'Failed to send ' + failedTypes.join(' and ') + ' reminder(s). Please try again.';
            }
            
            // Show notification modal
            showReminderNotification(message, success);
        })
        .catch(error => {
            console.error('Error:', error);
            showReminderNotification('An error occurred while sending reminders. Please try again.', false);
        })
        .finally(() => {
            if (submitBtn) {
                submitBtn.disabled = false;
                submitBtn.innerHTML = 'Send Reminder';
            }
        });
});

// ... existing code ...
</script>

<!-- Unpaid Receipt Templates -->
<?php foreach ($credits_by_customer as $cid => $group): ?>
<template id="unpaid-receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-balance="<?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?>">
    <div style="width:80mm; font-family: Arial, Helvetica, sans-serif; font-size:13px; color:#111; font-weight:700;">
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.2rem; font-weight:700; color:#111; margin-bottom:4px; letter-spacing:1px; font-family: Arial, Helvetica, sans-serif;">
                <h1 style="font-size:1.2rem; font-weight:700; color:#111; margin-bottom:4px; letter-spacing:1px; font-family: Arial, Helvetica, sans-serif;">UNPAID CREDIT ITEMS</h1>
                <h1 style="font-size:1.2rem; font-weight:700; color:#111; margin-bottom:4px; letter-spacing:1px; font-family: Arial, Helvetica, sans-serif;">RECEIPT</h1>
            </div>
            <?php if ($logoData): ?>
                <?php
                $mimeType = 'image/jpeg'; // Default to JPEG
                if (strpos(strtolower($logoName), '.png') !== false) {
                    $mimeType = 'image/png';
                } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                    $mimeType = 'image/gif';
                }
                ?>
                <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
            <?php endif; ?>
        </div>
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.1rem; font-weight:700; color:#111; font-family: Arial, Helvetica, sans-serif;">POS</div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Phone: <?php echo htmlspecialchars($storePhone); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Email: <?php echo htmlspecialchars($storeEmail); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
        </div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Date:</b> <span class="receipt-date"></span></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Receipt No:</b> <span class="receipt-number"></span></div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-weight:700; margin-bottom:4px; font-family: Arial, Helvetica, sans-serif;">Unpaid Credit Items:</div>
        <table style="width:100%; font-size:12px; border-collapse:collapse; margin-bottom:10px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <thead>
                <tr style="background:#f1f5f9; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Date</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Invoice</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Items</th>
                    <th style="text-align:center; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Qty</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Payment Details</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Waived</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Credit Amount</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Paid</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Balance</th>
                </tr>
            </thead>
            <tbody>
                <?php $rowAlt = false; foreach ($group['transactions'] as $credit): ?>
                <?php $item_balance = $credit['amount'] - $credit['amount_paid']; if ($item_balance <= 0.001) continue; ?>
                <tr style="background:<?php echo $rowAlt ? '#f8fafc' : '#fff'; ?>; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo date('M d', strtotime($credit['sale_date'])); ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['items'] ?? 'No items found'); ?></td>
                    <td style="text-align:center; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $credit['total_quantity'] ?? 0; ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                        <?php
                        $paymentDetails = '';
                        if (!empty($credit['payment_methods'])) {
                            $methods = explode(', ', $credit['payment_methods']);
                            $amounts = explode(', ', $credit['payment_amounts']);
                            $details = [];
                            for ($i = 0; $i < count($methods); $i++) {
                                if (isset($amounts[$i]) && $amounts[$i] > 0) {
                                    $details[] = ucfirst($methods[$i]) . ': ' . $currency . ' ' . number_format($amounts[$i], 2);
                                }
                            }
                            $paymentDetails = implode(', ', $details);
                        } elseif (!empty($credit['sale_payment_method']) && $credit['sale_paid_amount'] > 0) {
                            $paymentDetails = ucfirst($credit['sale_payment_method']) . ': ' . $currency . ' ' . number_format($credit['sale_paid_amount'], 2);
                        } else {
                            $paymentDetails = 'Credit Sale';
                        }
                        echo $paymentDetails;
                        ?>
                    </td>
                    <td style="text-align:right; padding:5px 4px; color:#dc2626; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($item_balance, 2); ?></td>
                </tr>
                <?php $rowAlt = !$rowAlt; endforeach; ?>
            </tbody>
        </table>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Balance:</b> <?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></div>
        <div style="margin:14px 0 6px 0; text-align:center; font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
    </div>
</template>
<?php endforeach; ?>

<!-- Add Credit Item Modal -->
<div class="modal fade" id="addCreditItemModal" tabindex="-1">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add Credit Item for <span id="addCreditCustomerName"></span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form id="addCreditItemForm" method="POST">
        <input type="hidden" name="add_credit_customer_id" id="addCreditCustomerId">
        <div class="modal-body">
          <div class="mb-3">
            <label class="form-label">Product</label>
            <select name="product_id" id="addCreditProduct" class="form-select" required>
              <option value="">Select Product</option>
              <?php foreach ($products as $product): ?>
                <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                  <?php echo htmlspecialchars($product['name']); ?> (<?php echo $currency . ' ' . number_format($product['price'], 2); ?>)
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="mb-3">
            <label class="form-label">Quantity</label>
            <input type="number" name="quantity" id="addCreditQuantity" class="form-control" min="1" value="1" required>
          </div>
          <div class="mb-3">
            <label class="form-label">Price</label>
            <input type="number" name="price" id="addCreditPrice" class="form-control" min="0" step="0.01" required>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-dark">Add Credit Item</button>
        </div>
      </form>
    </div>
  </div>
</div>
<script>
document.querySelectorAll('.add-credit-item-btn').forEach(function(btn) {
  btn.addEventListener('click', function() {
    var customerId = this.getAttribute('data-customer-id');
    var customerName = this.getAttribute('data-customer-name');
    document.getElementById('addCreditCustomerId').value = customerId;
    document.getElementById('addCreditCustomerName').textContent = customerName;
    document.getElementById('addCreditProduct').selectedIndex = 0;
    document.getElementById('addCreditQuantity').value = 1;
    document.getElementById('addCreditPrice').value = '';
    var modal = new bootstrap.Modal(document.getElementById('addCreditItemModal'));
    modal.show();
  });
});
document.getElementById('addCreditProduct').addEventListener('change', function() {
  var price = this.options[this.selectedIndex].getAttribute('data-price');
  document.getElementById('addCreditPrice').value = price || '';
});
document.getElementById('addCreditItemForm').addEventListener('submit', function(e) {
  e.preventDefault();
  var form = this;
  var formData = new FormData(form);
  fetch('', {
    method: 'POST',
    body: formData
  })
  .then(response => response.json())
  .then(data => {
    if (data.success) {
      alert('Credit item added successfully!');
      location.reload();
    } else {
      alert('Error: ' + data.message);
    }
  })
  .catch(err => alert('Error: ' + err));
  var modal = bootstrap.Modal.getInstance(document.getElementById('addCreditItemModal'));
  modal.hide();
});
</script>

<!-- Initialize print buttons when document is ready -->
<script>
// Add a print lock variable
let isPrinting = false;

document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing print buttons...'); // Debug log
    var printButtons = document.querySelectorAll('.print-credit-btn');
    console.log('Found print buttons:', printButtons.length); // Debug log
    
    printButtons.forEach(function(btn) {
        console.log('Adding click listener to button for customer:', btn.getAttribute('data-customer-id'));
    btn.addEventListener('click', function(e) {
        e.preventDefault();
            console.log('Print button clicked for customer: ' + this.getAttribute('data-customer-id'));
        
        if (isPrinting) {
            console.log('Print already in progress, please wait...');
            return;
        }
        
        try {
            isPrinting = true;
            var cid = this.getAttribute('data-customer-id');
            console.log('Print button clicked for customer:', cid);
            
            var template = document.getElementById('unpaid-receipt-template-' + cid);
            if (!template) {
                    throw new Error('Receipt template not found for customer: ' + cid);
            }

            // Extract only the <tr> rows from the template's table (skip any non-<tr> content)
            var tempDiv = document.createElement('div');
            tempDiv.innerHTML = template.innerHTML;
            var trRows = '';
            var table = tempDiv.querySelector('table');
            if (table) {
                var trs = table.querySelectorAll('tbody tr');
                var currentDate = '';
                var currentInvoice = '';
                var currentSection = '';
                
                trs.forEach(function(tr) {
                    var cells = tr.querySelectorAll('td');
                    if (cells.length >= 9) {
                        var date = cells[0].textContent.trim();
                        var invoice = cells[1].textContent.trim();
                        
                        // Start new section if date or invoice changes
                        if (date !== currentDate || invoice !== currentInvoice) {
                            if (currentSection !== '') {
                                trRows += currentSection + '</table></div>';
                            }
                            currentDate = date;
                            currentInvoice = invoice;
                            currentSection = '<div style="margin-bottom: 15px;"><div style="font-weight: bold; margin-bottom: 5px;">Date: ' + date + ' | Invoice: ' + invoice + '</div><table><thead><tr><th>Items</th><th>Qty</th><th>Payment Details</th><th class="text-right">Waived</th><th class="text-right">Credit Amount</th><th class="text-right">Paid</th><th class="text-right">Balance</th></tr></thead><tbody>';
                        }
                        
                        // Add the row without date and invoice columns
                        currentSection += '<tr><td>' + cells[2].textContent + '</td><td>' + cells[3].textContent + '</td><td>' + cells[4].textContent + '</td><td class="text-right">' + cells[5].textContent + '</td><td class="text-right">' + cells[6].textContent + '</td><td class="text-right">' + cells[7].textContent + '</td><td class="text-right">' + cells[8].textContent + '</td></tr>';
                    }
                });
                
                // Close the last section
                if (currentSection !== '') {
                    trRows += currentSection + '</table></div>';
                }
            }

            var printWindow = window.open('', '_blank', 'width=400,height=600');
            if (!printWindow) {
                throw new Error('Pop-up blocked. Please allow pop-ups for this site.');
            }

            // Write the receipt content with only one set of store info and receipt info
            printWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Unpaid Credit Receipt</title>
                    <meta charset="UTF-8">
                    <style>
                        @media print {
                            @page {
                                size: 80mm auto;
                                margin: 0;
                            }
                            body {
                                width: 80mm;
                                margin: 0;
                                padding: 5px;
                            }
                            .no-print {
                                display: none !important;
                            }
                        }
                        body { 
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            color: #000;
                            margin: 0;
                            padding: 5px;
                            width: 80mm;
                            max-width: 80mm;
                        }
                        .store-info {
                            text-align: center;
                            margin-bottom: 8px;
                        }
                        .store-name {
                            font-size: 1.2rem;
                            font-weight: 700;
                            margin-bottom: 2px;
                        }
                        .divider {
                            border-top: 1px dashed #000;
                            margin: 8px 0;
                        }
                        .section-title {
                            font-weight: bold;
                            margin: 8px 0;
                            text-align: center;
                        }
                        table {
                            width: 100%;
                            border-collapse: collapse;
                            font-size: 11px;
                            margin-bottom: 10px;
                        }
                        th, td {
                            padding: 2px;
                            text-align: left;
                            border-bottom: 1px solid #ddd;
                        }
                        th {
                            border-bottom: 1px solid #000;
                        }
                        .text-right {
                            text-align: right;
                        }
                        .qr-code {
                            text-align: center;
                            margin: 10px 0;
                        }
                        .qr-code img {
                            width: 120px;
                            height: 120px;
                        }
                        .footer {
                            text-align: center;
                            margin-top: 10px;
                            font-size: 11px;
                        }
                    </style>
                </head>
                <body>
                    <div class="store-info">
                        <div style="font-size:1.1rem; font-weight:900; color:#000; margin-bottom:2px;">UNPAID CREDIT ITEMS <br> RECEIPT</div>
                        <?php if ($logoData): ?>
                            <?php
                            $mimeType = 'image/jpeg'; // Default to JPEG
                            if (strpos(strtolower($logoName), '.png') !== false) {
                                $mimeType = 'image/png';
                            } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                                $mimeType = 'image/gif';
                            }
                            ?>
                            <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
                        <?php endif; ?>
                        <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
                        <div><?php echo htmlspecialchars($storeAddress); ?></div>
                        <div>Phone: <?php echo htmlspecialchars($storePhone); ?></div>
                        <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                        <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                    </div>
                    <div class="divider"></div>
                    <div class="section-title">UNPAID CREDIT ITEMS</div>
                    <div class="divider"></div>
                    <div><b>Customer:</b> <span class="customer-name"></span></div>
                    <div><b>Date:</b> <span class="receipt-date"></span></div>
                    <div><b>Receipt No:</b> <span class="receipt-number"></span></div>
                    <div class="divider"></div>
                    <div id="invoice-sections">
                        ${trRows}
                    </div>
                    <div class="divider"></div>
                    <div><b>Total Balance:</b> <span class="total-balance"></span></div>
                    <div class="divider"></div>
                    <div style="text-align:center; margin:12px 0;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
                    <div class="qr-code">
                        <img id="receipt-qr" src="" alt="Receipt QR Code">
                    </div>
                    <div class="divider"></div>
                    <div class="footer">Designed and developed by Vicotechnologies.co.ke <br>+254702214793</div>
                </body>
                </html>
            `);

            printWindow.document.close();

            printWindow.onload = function() {
                try {
                    var now = new Date();
                    var dateStr = now.getFullYear() + '-' + String(now.getMonth()+1).padStart(2,'0') + '-' + String(now.getDate()).padStart(2,'0') + ' ' + String(now.getHours()).padStart(2,'0') + ':' + String(now.getMinutes()).padStart(2,'0') + ':' + String(now.getSeconds()).padStart(2,'0');
                    var receiptNo = 'UNPAID-' + cid + '-' + now.getTime();
                    
                    // Update receipt elements
                    printWindow.document.querySelector('.customer-name').textContent = template.getAttribute('data-customer-name');
                    printWindow.document.querySelector('.receipt-date').textContent = dateStr;
                    printWindow.document.querySelector('.receipt-number').textContent = receiptNo;
                    printWindow.document.querySelector('.total-balance').textContent = template.getAttribute('data-total-balance');

                    // Generate QR code data
                    var receiptInfo = {
                        store: '<?php echo addslashes($storeName); ?>',
                        address: '<?php echo addslashes($storeAddress); ?>',
                        phone: '<?php echo addslashes($storePhone); ?>',
                        email: '<?php echo addslashes($storeEmail); ?>',
                        website: '<?php echo addslashes($storeWebsite); ?>',
                        date: dateStr,
                        receipt_no: receiptNo,
                        customer: template.getAttribute('data-customer-name'),
                        total_balance: template.getAttribute('data-total-balance'),
                        type: 'Unpaid Credit Items'
                    };

                    // Add items to QR data
                    receiptInfo.items = [];
                    var sections = printWindow.document.querySelectorAll('#invoice-sections > div');
                    sections.forEach(function(section) {
                        var dateInvoiceDiv = section.querySelector('div');
                        var dateInvoiceText = dateInvoiceDiv.textContent;
                        var dateMatch = dateInvoiceText.match(/Date: (.+?) \|/);
                        var invoiceMatch = dateInvoiceText.match(/Invoice: (.+)/);
                        
                        var date = dateMatch ? dateMatch[1].trim() : '';
                        var invoice = invoiceMatch ? invoiceMatch[1].trim() : '';
                        
                        var rows = section.querySelectorAll('table tbody tr');
                        rows.forEach(function(row) {
                            var cells = row.querySelectorAll('td');
                            if (cells.length === 7) {
                                receiptInfo.items.push({
                                    date: date,
                                    invoice: invoice,
                                    items: cells[0].textContent,
                                    qty: cells[1].textContent,
                                    payment_details: cells[2].textContent,
                                    waived: cells[3].textContent,
                                    credit_amount: cells[4].textContent,
                                    paid: cells[5].textContent,
                                    balance: cells[6].textContent
                                });
                            }
                        });
                    });

                    // Generate QR code using qrserver
                    var qrImg = printWindow.document.getElementById('receipt-qr');
                    qrImg.src = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(JSON.stringify(receiptInfo));

                    // Wait for QR code to load before printing
                    qrImg.onload = function() {
                        setTimeout(function() {
                            printWindow.print();
                            setTimeout(function() {
                                printWindow.close();
                                isPrinting = false;
                            }, 1000);
                        }, 500);
                    };

                    // Handle QR code load error
                    qrImg.onerror = function() {
                        console.error('QR code failed to load');
                        setTimeout(function() {
                            printWindow.print();
                            setTimeout(function() {
                                printWindow.close();
                                isPrinting = false;
                            }, 1000);
                        }, 500);
                    };

                } catch (error) {
                    console.error('Error preparing receipt:', error);
                    alert('Error preparing receipt. Please try again.');
                    isPrinting = false;
                }
            };

        } catch (error) {
            console.error('Print button error:', error);
            alert(error.message || 'Error printing receipt. Please try again.');
            isPrinting = false;
        }
    });
    });
});
</script>
</body>
</html> 
_file>
rewritten_file>